/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.qa.unittest;

import com.zimbra.common.mime.MimeMessage;
import com.zimbra.common.service.ServiceException;
import com.zimbra.common.soap.Element;
import com.zimbra.common.soap.MailConstants;
import com.zimbra.common.soap.SoapFaultException;
import com.zimbra.common.util.ByteUtil;
import com.zimbra.cs.account.Account;
import com.zimbra.cs.account.Config;
import com.zimbra.cs.account.Provisioning;
import com.zimbra.cs.filter.FilterUtil;
import com.zimbra.cs.filter.RuleManager;
import com.zimbra.cs.filter.RuleRewriter;
import com.zimbra.cs.filter.SieveToSoap;
import com.zimbra.cs.filter.SoapToSieve;
import com.zimbra.cs.zclient.ZFilterAction;
import com.zimbra.cs.zclient.ZFilterCondition;
import com.zimbra.cs.zclient.ZFilterRule;
import com.zimbra.cs.zclient.ZFilterRules;
import com.zimbra.cs.zclient.ZFolder;
import com.zimbra.cs.zclient.ZMailbox;
import com.zimbra.cs.zclient.ZMessage;
import com.zimbra.cs.zclient.ZTag;
import com.zimbra.qa.unittest.MessageBuilder;
import com.zimbra.qa.unittest.TestUtil;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import junit.framework.TestCase;
import org.apache.jsieve.parser.generated.Node;
import org.apache.jsieve.parser.generated.ParseException;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class TestFilter
extends TestCase {
    private static String USER_NAME = "user1";
    private static String REMOTE_USER_NAME = "user2";
    private static String NAME_PREFIX = "TestFilter";
    private static String TAG_TEST_BASE64_SUBJECT = NAME_PREFIX + "-testBase64Subject";
    private static String FOLDER1_NAME = NAME_PREFIX + "-folder1";
    private static String FOLDER2_NAME = NAME_PREFIX + "-folder2";
    private static String FOLDER1_PATH = "/" + FOLDER1_NAME;
    private static String FOLDER2_PATH = "/" + FOLDER2_NAME;
    private static String TAG1_NAME = NAME_PREFIX + "-tag1";
    private static String TAG2_NAME = NAME_PREFIX + "-tag2";
    private static String MOUNTPOINT_FOLDER_NAME = NAME_PREFIX + " mountpoint";
    private static String MOUNTPOINT_SUBFOLDER_NAME = NAME_PREFIX + " mountpoint subfolder";
    private static String MOUNTPOINT_SUBFOLDER_PATH = "/" + MOUNTPOINT_FOLDER_NAME + "/" + MOUNTPOINT_SUBFOLDER_NAME;
    private ZMailbox mMbox;
    private ZFilterRules mOriginalRules;
    private String mOriginalSpamApplyUserFilters;

    public void setUp() throws Exception {
        this.mMbox = TestUtil.getZMailbox(USER_NAME);
        this.cleanUp();
        ZMailbox remoteMbox = TestUtil.getZMailbox(REMOTE_USER_NAME);
        TestUtil.createMountpoint(remoteMbox, "/" + MOUNTPOINT_FOLDER_NAME, this.mMbox, MOUNTPOINT_FOLDER_NAME);
        TestUtil.createFolder(remoteMbox, MOUNTPOINT_SUBFOLDER_PATH);
        this.mOriginalRules = this.mMbox.getFilterRules();
        this.saveRules(this.mMbox, this.getRules());
        Account account = TestUtil.getAccount(USER_NAME);
        this.mOriginalSpamApplyUserFilters = account.getAttr("zimbraSpamApplyUserFilters");
    }

    public void testQuoteEscape() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        String folderName = NAME_PREFIX + " testQuoteEscape";
        TestUtil.createFolder(this.mMbox, folderName);
        ZFilterCondition.ZHeaderCondition condition = new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "a \" b");
        ZFilterAction.ZFileIntoAction action = new ZFilterAction.ZFileIntoAction(folderName);
        conditions.add(condition);
        actions.add(action);
        rules.add(new ZFilterRule(folderName, true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " a \" b y z";
        String searchSubject = NAME_PREFIX + " a b y z";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.getMessage(this.mMbox, "in:\"" + folderName + "\" subject:\"" + searchSubject + "\"");
    }

    public void testBackslashEscape() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        String folderName = NAME_PREFIX + " testBackslashEscape";
        TestUtil.createFolder(this.mMbox, folderName);
        ZFilterCondition.ZHeaderCondition condition = new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "a \\ b");
        ZFilterAction.ZFileIntoAction action = new ZFilterAction.ZFileIntoAction(folderName);
        conditions.add(condition);
        actions.add(action);
        rules.add(new ZFilterRule(folderName, true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " a \\ b y z";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.getMessage(this.mMbox, "in:\"" + folderName + "\" subject:\"" + subject + "\"");
    }

    public void testBase64Subject() throws Exception {
        String address = TestUtil.getAddress(USER_NAME);
        TestUtil.addMessageLmtp("=?UTF-8?B?W2l0dnNmLUluY2lkZW5jaWFzXVs0OTc3Ml0gW2luY2lkZW5jaWFzLXZpbGxhbnVldmFdIENvcnRlcyBkZSBsdXosIGTDrWEgMjUvMDkvMjAwNi4=?=", address, address);
        List<ZMessage> messages = TestUtil.search(this.mMbox, "Cortes de luz");
        TestFilter.assertEquals((String)"Unexpected number of messages", (int)1, (int)messages.size());
        List<ZTag> tags = this.mMbox.getTags(messages.get(0).getTagIds());
        TestFilter.assertEquals((String)"Unexpected number of tags", (int)1, (int)tags.size());
        TestFilter.assertEquals((String)"Tag didn't match", (String)TAG_TEST_BASE64_SUBJECT, (String)tags.get(0).getName());
    }

    public void testMatchMultipleFilters() throws Exception {
        String sender = TestUtil.getAddress("multiplefilters");
        String recipient = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " This goes into folder1 and folder2";
        TestUtil.addMessageLmtp(subject, recipient, sender);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:" + FOLDER1_PATH + " " + subject);
        TestUtil.verifyTag(this.mMbox, msg, TAG1_NAME);
        TestUtil.verifyTag(this.mMbox, msg, TAG2_NAME);
        TestUtil.verifyFlag(this.mMbox, msg, ZMessage.Flag.flagged);
        msg = TestUtil.getMessage(this.mMbox, "in:" + FOLDER2_PATH + " " + subject);
        TestUtil.verifyTag(this.mMbox, msg, TAG1_NAME);
        TestUtil.verifyTag(this.mMbox, msg, TAG2_NAME);
        TestUtil.verifyFlag(this.mMbox, msg, ZMessage.Flag.flagged);
    }

    public void testSpam() throws Exception {
        ZMailbox mbox = TestUtil.getZMailbox(USER_NAME);
        String sender = TestUtil.getAddress(USER_NAME);
        String[] recipients = new String[]{sender};
        String message = TestUtil.getTestMessage(NAME_PREFIX + " testSpam", USER_NAME, USER_NAME, null);
        Config config = Provisioning.getInstance().getConfig();
        message = config.getAttr("zimbraSpamHeader") + ": " + config.getAttr("zimbraSpamHeaderValue") + "\r\n" + message;
        TestFilter.assertEquals((int)0, (int)TestUtil.search(mbox, "in:junk subject:testSpam").size());
        TestUtil.addMessageLmtp(recipients, sender, message);
        ZMessage msg = TestUtil.getMessage(mbox, "in:junk subject:testSpam");
        mbox.deleteMessage(msg.getId());
        ZFilterRules rules = this.getRules();
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "testSpam"));
        actions.add(new ZFilterAction.ZFileIntoAction(FOLDER1_PATH));
        rules.getRules().add(new ZFilterRule("testBug5455", true, false, conditions, actions));
        this.saveRules(this.mMbox, rules);
        TestUtil.setAccountAttr(USER_NAME, "zimbraSpamApplyUserFilters", "TRUE");
        TestUtil.addMessageLmtp(recipients, sender, message);
        msg = TestUtil.waitForMessage(mbox, "in:" + FOLDER1_PATH + " subject:testSpam");
        mbox.deleteMessage(msg.getId());
        TestUtil.setAccountAttr(USER_NAME, "zimbraSpamApplyUserFilters", "FALSE");
        TestUtil.addMessageLmtp(recipients, sender, message);
        TestUtil.waitForMessage(mbox, "in:junk subject:testSpam");
    }

    public void testBug5455() throws Exception {
        String recipient = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " Testing bug5455";
        TestUtil.addMessageLmtp(subject, recipient, recipient);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:" + FOLDER1_PATH + " " + subject);
        TestUtil.verifyFlag(this.mMbox, msg, ZMessage.Flag.flagged);
        TestUtil.verifyTag(this.mMbox, msg, TAG1_NAME);
        msg = TestUtil.getMessage(this.mMbox, "in:" + FOLDER2_PATH + " " + subject);
        TestUtil.verifyFlag(this.mMbox, msg, ZMessage.Flag.flagged);
        TestUtil.verifyTag(this.mMbox, msg, TAG1_NAME);
    }

    public void testDiscard() throws Exception {
        String recipient = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " doDiscard";
        TestUtil.addMessageLmtp(subject, recipient, recipient);
        subject = NAME_PREFIX + " dontDiscard";
        TestUtil.addMessageLmtp(subject, recipient, recipient);
        TestUtil.waitForMessage(this.mMbox, "in:inbox dontDiscard");
        TestFilter.assertEquals((int)0, (int)TestUtil.search(this.mMbox, "in:inbox doDiscard").size());
    }

    public void testMountpoint() throws Exception {
        ZMailbox mbox = TestUtil.getZMailbox(USER_NAME);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " mountpointRoot1";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.waitForMessage(mbox, "in:\"/" + MOUNTPOINT_FOLDER_NAME + "\" subject:\"" + subject + "\"");
        TestFilter.assertEquals((int)0, (int)TestUtil.search(mbox, "in:inbox subject:\"" + subject + "\"").size());
        ZMailbox remote = TestUtil.getZMailbox(REMOTE_USER_NAME);
        ZFolder remoteFolder = remote.getFolderByPath("/" + MOUNTPOINT_FOLDER_NAME);
        remote.deleteFolder(remoteFolder.getId());
        subject = NAME_PREFIX + " mountpointRoot2";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.waitForMessage(mbox, "in:inbox subject:\"" + subject + "\"");
        TestFilter.assertEquals((int)0, (int)TestUtil.search(mbox, "in:\"/" + MOUNTPOINT_FOLDER_NAME + "\" subject:\"" + subject + "\"").size());
    }

    public void testMountpointSubfolder() throws Exception {
        ZMailbox mbox = TestUtil.getZMailbox(USER_NAME);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " mountpointSub1";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.waitForMessage(mbox, "in:\"" + MOUNTPOINT_SUBFOLDER_PATH + "\" subject:\"" + subject + "\"");
        TestFilter.assertEquals((int)0, (int)TestUtil.search(mbox, "in:inbox subject:\"" + subject + "\"").size());
        ZMailbox remote = TestUtil.getZMailbox(REMOTE_USER_NAME);
        ZFolder remoteFolder = remote.getFolderByPath(MOUNTPOINT_SUBFOLDER_PATH);
        remote.deleteFolder(remoteFolder.getId());
        subject = NAME_PREFIX + " mountpointSub2";
        TestUtil.addMessageLmtp(subject, address, address);
        TestUtil.waitForMessage(mbox, "in:inbox subject:\"" + subject + "\"");
        TestFilter.assertEquals((int)0, (int)TestUtil.search(mbox, "in:\"" + MOUNTPOINT_SUBFOLDER_PATH + "\" subject:\"" + subject + "\"").size());
    }

    public void testEscape() {
        this.doTestEscape("Hello, \"Dave\"", "Hello, \\\"Dave\\\"");
        this.doTestEscape("\\/\\/", "\\\\/\\\\/");
        this.doTestEscape("\"\\\"", "\\\"\\\\\\\"");
    }

    private void doTestEscape(String original, String escaped) {
        TestFilter.assertEquals((String)escaped, (String)FilterUtil.escape(original));
        TestFilter.assertEquals((String)original, (String)FilterUtil.unescape(escaped));
    }

    public void testConversion() throws Exception {
        String scriptPath = "/opt/zimbra/unittest/test.sieve";
        String script = new String(ByteUtil.getContent(new File(scriptPath)));
        TestFilter.assertNotNull((Object)script);
        TestFilter.assertTrue((script.length() > 0 ? 1 : 0) != 0);
        String convertedScript = this.normalize(script);
        script = this.normalizeWhiteSpace(script);
        convertedScript = this.normalizeWhiteSpace(convertedScript);
        script = script.replace("fileInto", "fileinto");
        TestFilter.assertEquals((String)script, (String)convertedScript);
    }

    public void testMarkRead() throws Exception {
        String folderName = NAME_PREFIX + " testMarkRead";
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "testMarkRead"));
        actions.add(new ZFilterAction.ZFileIntoAction(folderName));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.READ));
        rules.add(new ZFilterRule("testMarkRead", true, false, conditions, actions));
        this.saveRules(this.mMbox, new ZFilterRules(rules));
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " testMarkRead";
        TestUtil.addMessageLmtp(subject, address, address);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:\"" + folderName + "\" subject:\"" + subject + "\"");
        String flags = msg.getFlags();
        TestFilter.assertTrue((String)("Unexpected flags: " + flags), (flags == null || flags.indexOf(ZMessage.Flag.unread.getFlagChar()) < 0 ? 1 : 0) != 0);
    }

    public void testHeaderFolding() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "a b c"));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.FLAGGED));
        rules.add(new ZFilterRule("testHeaderFolding", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " a b\r\n c";
        TestUtil.addMessageLmtp(subject, address, address);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:inbox subject:\"a b c\"");
        TestFilter.assertTrue((String)"Message was not flagged", (boolean)msg.isFlagged());
    }

    public void testManyAsterisks() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        ZFilterCondition.ZHeaderCondition condition = new ZFilterCondition.ZHeaderCondition("from", ZFilterCondition.HeaderOp.MATCHES, "*****address@yahoo.com");
        ZFilterAction.ZKeepAction action = new ZFilterAction.ZKeepAction();
        conditions.add(condition);
        actions.add(action);
        rules.add(new ZFilterRule("test many asterisks", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        try {
            this.mMbox.saveFilterRules(zRules);
            TestFilter.fail((String)"Saving filter rules with quotes should not have succeeded");
        }
        catch (SoapFaultException e) {
            TestFilter.assertTrue((String)("Unexpected exception: " + e), (boolean)e.getMessage().contains("four asterisks"));
        }
    }

    public void testHeaderMatches() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        String pattern = NAME_PREFIX.substring(0, 4) + "*";
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.MATCHES, pattern));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.FLAGGED));
        rules.add(new ZFilterRule("testHeaderMatches", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " testHeaderMatches";
        TestUtil.addMessageLmtp(subject, address, address);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:inbox subject:testHeaderMatches");
        TestFilter.assertTrue((String)"Message was not flagged", (boolean)msg.isFlagged());
    }

    public void testBodyContains() throws Exception {
        this.doBodyContainsTest("text version of the main body", true);
        this.doBodyContainsTest("HTML version of the main body", true);
        this.doBodyContainsTest("text attachment", false);
        this.doBodyContainsTest("HTML attachment", false);
        this.doBodyContainsTest("text version of the attached message body", true);
        this.doBodyContainsTest("HTML version the attached message body", true);
        this.doBodyContainsTest("body of a plain attached message", true);
        this.doBodyContainsTest("TeXt VeRsIoN", true);
        this.doBodyContainsTest("hTmL vErSiOn", true);
        this.doBodyContainsTest("This is the text version of the main body of the message.", true);
        this.doBodyContainsTest("This is the text version of the main body of the message. This is the second line.", true);
    }

    private void doBodyContainsTest(String substring, boolean contains) throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        conditions.add(new ZFilterCondition.ZBodyCondition(ZFilterCondition.BodyOp.CONTAINS, substring));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.FLAGGED));
        rules.add(new ZFilterRule("testBodyContains", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String msgContent = new String(ByteUtil.getContent(new File("/opt/zimbra/unittest/TestFilter-testBodyContains.msg")));
        TestUtil.addMessageLmtp(new String[]{address}, address, msgContent);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:inbox subject:testBodyContains");
        TestFilter.assertEquals((String)"Unexpected message flag state", (boolean)contains, (boolean)msg.isFlagged());
        this.mMbox.deleteMessage(msg.getId());
    }

    public void testRedirect() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "testRedirect"));
        actions.add(new ZFilterAction.ZRedirectAction(TestUtil.getAddress(REMOTE_USER_NAME)));
        rules.add(new ZFilterRule("testRedirect", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " testRedirect";
        TestUtil.addMessageLmtp(subject, address, address);
        List<ZMessage> messages = TestUtil.search(this.mMbox, "subject:\"" + subject + "\"");
        TestFilter.assertEquals((int)0, (int)messages.size());
        ZMailbox remoteMbox = TestUtil.getZMailbox(REMOTE_USER_NAME);
        ZMessage msg = TestUtil.waitForMessage(remoteMbox, "in:inbox subject:\"" + subject + "\"");
        byte[] content = TestUtil.getContent(remoteMbox, msg.getId()).getBytes();
        MimeMessage mimeMsg = new MimeMessage(new ByteArrayInputStream(content));
        Account user1 = TestUtil.getAccount(USER_NAME);
        TestFilter.assertEquals((String)user1.getName(), (String)mimeMsg.getHeader("X-Zimbra-Forwarded"));
    }

    public void testRedirectMailLoop() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "testRedirectMailLoop"));
        actions.add(new ZFilterAction.ZRedirectAction(TestUtil.getAddress(USER_NAME)));
        rules.add(new ZFilterRule("testRedirectMailLoop", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String subject = NAME_PREFIX + " testRedirectMailLoop";
        TestUtil.addMessageLmtp(subject, USER_NAME, USER_NAME);
        ZMessage msg = TestUtil.waitForMessage(this.mMbox, "subject:\"" + subject + "\"");
        byte[] content = TestUtil.getContent(this.mMbox, msg.getId()).getBytes();
        MimeMessage mimeMsg = new MimeMessage(new ByteArrayInputStream(content));
        Account user1 = TestUtil.getAccount(USER_NAME);
        TestFilter.assertEquals((String)user1.getName(), (String)mimeMsg.getHeader("X-Zimbra-Forwarded"));
    }

    public void testAttachment() throws Exception {
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        ZTag tag1 = this.mMbox.createTag(NAME_PREFIX + " testAttachment1", null);
        ZTag tag2 = this.mMbox.createTag(NAME_PREFIX + " testAttachment2", null);
        conditions.add(new ZFilterCondition.ZAttachmentExistsCondition(true));
        actions.add(new ZFilterAction.ZTagAction(tag1.getName()));
        rules.add(new ZFilterRule("testAttachment1", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZAttachmentExistsCondition(false));
        actions.add(new ZFilterAction.ZTagAction(tag2.getName()));
        rules.add(new ZFilterRule("testAttachment2", true, false, conditions, actions));
        ZFilterRules zRules = new ZFilterRules(rules);
        this.saveRules(this.mMbox, zRules);
        String address = TestUtil.getAddress(USER_NAME);
        String subject = NAME_PREFIX + " testAttachment1";
        String msgContent = new MessageBuilder().withSubject(subject).withAttachment("attach this", "attach.txt", "text/plain").create();
        TestUtil.addMessageLmtp(new String[]{address}, address, msgContent);
        ZMessage msg = TestUtil.getMessage(this.mMbox, "in:inbox subject:\"" + subject + "\"");
        Set<String> tagIds = this.getTagIds(msg);
        TestFilter.assertEquals((int)1, (int)tagIds.size());
        TestFilter.assertTrue((boolean)tagIds.contains(tag1.getId()));
        subject = NAME_PREFIX + " testAttachment2";
        msgContent = new MessageBuilder().withSubject(subject).create();
        TestUtil.addMessageLmtp(new String[]{address}, address, msgContent);
        msg = TestUtil.getMessage(this.mMbox, "in:inbox subject:\"" + subject + "\"");
        tagIds = this.getTagIds(msg);
        TestFilter.assertEquals((int)1, (int)tagIds.size());
        TestFilter.assertTrue((boolean)tagIds.contains(tag2.getId()));
    }

    private Set<String> getTagIds(ZMessage msg) {
        HashSet<String> ids = new HashSet<String>();
        for (String id : msg.getTagIds().split(",")) {
            ids.add(id);
        }
        return ids;
    }

    public void testParseSize() throws Exception {
        TestFilter.assertEquals((int)10, (int)FilterUtil.parseSize("10"));
        TestFilter.assertEquals((int)10240, (int)FilterUtil.parseSize("10k"));
        TestFilter.assertEquals((int)10240, (int)FilterUtil.parseSize("10K"));
        TestFilter.assertEquals((int)0xA00000, (int)FilterUtil.parseSize("10M"));
        TestFilter.assertEquals((int)Integer.MIN_VALUE, (int)FilterUtil.parseSize("10G"));
        try {
            FilterUtil.parseSize("10Q");
            TestFilter.fail((String)"Should not have parsed bogus size value.");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    public void testOldApi() throws Exception {
        ZMailbox mbox = TestUtil.getZMailbox(USER_NAME);
        Account account = TestUtil.getAccount(USER_NAME);
        String oldRules = account.getMailSieveScript();
        Element response = mbox.invoke(new Element.XMLElement(MailConstants.GET_RULES_REQUEST));
        Element rulesEl = response.getElement("rules").detach();
        RuleRewriter.sanitizeRules(rulesEl);
        Element request = rulesEl.getFactory().createElement(MailConstants.SAVE_RULES_REQUEST);
        request.addElement(rulesEl);
        response = mbox.invoke(request);
        account = TestUtil.getAccount(USER_NAME);
        oldRules = this.normalize(oldRules);
        String newRules = this.normalize(account.getMailSieveScript());
        TestFilter.assertEquals((String)oldRules, (String)newRules);
    }

    private String normalize(String script) throws ParseException, ServiceException {
        List<String> ruleNames = RuleManager.getRuleNames(script);
        Node node = RuleManager.getSieveFactory().parse((InputStream)new ByteArrayInputStream(script.getBytes()));
        SieveToSoap sieveToSoap = new SieveToSoap(Element.XMLElement.mFactory, ruleNames);
        sieveToSoap.accept(node);
        SoapToSieve soapToSieve = new SoapToSieve(sieveToSoap.getRootElement());
        return soapToSieve.getSieveScript();
    }

    public void testStripBracketsAndQuotes() throws Exception {
        TestFilter.assertEquals(null, (String)RuleRewriter.stripBracketsAndQuotes(null));
        TestFilter.assertEquals((String)"", (String)RuleRewriter.stripBracketsAndQuotes(""));
        TestFilter.assertEquals((String)"x", (String)RuleRewriter.stripBracketsAndQuotes("x"));
        TestFilter.assertEquals((String)"x", (String)RuleRewriter.stripBracketsAndQuotes("[\"x\"]"));
        TestFilter.assertEquals((String)"x", (String)RuleRewriter.stripBracketsAndQuotes("\"x\""));
        TestFilter.assertEquals((String)"x\"", (String)RuleRewriter.stripBracketsAndQuotes("x\""));
        TestFilter.assertEquals((String)"[\"x\"", (String)RuleRewriter.stripBracketsAndQuotes("[\"x\""));
    }

    private String normalizeWhiteSpace(String script) {
        StringBuilder buf = new StringBuilder(script.length());
        boolean inWhiteSpace = false;
        for (int i = 0; i < script.length(); ++i) {
            String c = script.substring(i, i + 1);
            if (c.matches("\\s") || c.equals("\\n")) {
                if (inWhiteSpace) continue;
                buf.append(" ");
                inWhiteSpace = true;
                continue;
            }
            inWhiteSpace = false;
            buf.append(c);
        }
        return buf.toString();
    }

    protected void tearDown() throws Exception {
        this.mMbox.saveFilterRules(this.mOriginalRules);
        TestUtil.setAccountAttr(USER_NAME, "zimbraSpamApplyUserFilters", this.mOriginalSpamApplyUserFilters);
        this.cleanUp();
    }

    private void cleanUp() throws Exception {
        TestUtil.deleteTestData(USER_NAME, NAME_PREFIX);
        TestUtil.deleteTestData(REMOTE_USER_NAME, NAME_PREFIX);
        for (ZMessage msg : TestUtil.search(this.mMbox, "cortes de luz")) {
            this.mMbox.deleteMessage(msg.getId());
        }
    }

    private ZFilterRules getRules() throws Exception {
        ArrayList<ZFilterRule> rules = new ArrayList<ZFilterRule>();
        ArrayList<ZFilterCondition> conditions = new ArrayList<ZFilterCondition>();
        ArrayList<ZFilterAction> actions = new ArrayList<ZFilterAction>();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "Cortes de luz"));
        actions.add(new ZFilterAction.ZTagAction(TAG_TEST_BASE64_SUBJECT));
        rules.add(new ZFilterRule("testBase64Subject", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "folder1"));
        actions.add(new ZFilterAction.ZFileIntoAction(FOLDER1_PATH));
        actions.add(new ZFilterAction.ZTagAction(TAG1_NAME));
        rules.add(new ZFilterRule("testMatchMultipleFilters1", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("from", ZFilterCondition.HeaderOp.CONTAINS, "multiplefilters"));
        actions.add(new ZFilterAction.ZFileIntoAction(FOLDER2_PATH));
        actions.add(new ZFilterAction.ZTagAction(TAG2_NAME));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.FLAGGED));
        rules.add(new ZFilterRule("testMatchMultipleFilters2", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "bug5455"));
        actions.add(new ZFilterAction.ZMarkAction(ZFilterAction.MarkOp.FLAGGED));
        actions.add(new ZFilterAction.ZFileIntoAction(FOLDER1_PATH));
        actions.add(new ZFilterAction.ZTagAction(TAG1_NAME));
        actions.add(new ZFilterAction.ZFileIntoAction(FOLDER2_PATH));
        rules.add(new ZFilterRule("testBug5455", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "doDiscard"));
        actions.add(new ZFilterAction.ZDiscardAction());
        rules.add(new ZFilterRule("testDiscard", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "mountpointRoot"));
        actions.add(new ZFilterAction.ZFileIntoAction("/" + MOUNTPOINT_FOLDER_NAME));
        rules.add(new ZFilterRule("testMountpoint", true, false, conditions, actions));
        conditions = new ArrayList();
        actions = new ArrayList();
        conditions.add(new ZFilterCondition.ZHeaderCondition("subject", ZFilterCondition.HeaderOp.CONTAINS, "mountpointSub"));
        actions.add(new ZFilterAction.ZFileIntoAction(MOUNTPOINT_SUBFOLDER_PATH));
        rules.add(new ZFilterRule("testMountpointSubfolder", true, false, conditions, actions));
        return new ZFilterRules(rules);
    }

    private void saveRules(ZMailbox mbox, ZFilterRules rules) throws Exception {
        Element.XMLElement root = new Element.XMLElement("test");
        Element expected = rules.toElement(root);
        mbox.saveFilterRules(rules);
        rules = mbox.getFilterRules(true);
        Element actual = rules.toElement(root);
        TestUtil.assertEquals(expected, actual);
    }

    public static void main(String[] args) throws Exception {
        TestUtil.cliSetup();
        TestUtil.runTest(TestFilter.class);
    }
}

