/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.perf.chart;

import com.zimbra.common.util.CsvReader;
import com.zimbra.common.util.Pair;
import com.zimbra.common.util.StringUtil;
import com.zimbra.perf.chart.ChartSettings;
import com.zimbra.perf.chart.GroupPlotSettings;
import com.zimbra.perf.chart.PlotSettings;
import com.zimbra.perf.chart.SummaryAnalyzer;
import com.zimbra.perf.chart.XMLChartConfig;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.LogarithmicAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.general.SeriesException;
import org.jfree.data.time.FixedMillisecond;
import org.jfree.data.time.MovingAverage;
import org.jfree.data.time.RegularTimePeriod;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class ChartUtil {
    private static final String OPT_HELP = "help";
    private static final String OPT_CONF = "conf";
    private static final String OPT_SRCDIR = "srcdir";
    private static final String OPT_DESTDIR = "destdir";
    private static final String OPT_TITLE = "title";
    private static final String OPT_START_AT = "start-at";
    private static final String OPT_END_AT = "end-at";
    private static final String OPT_AGGREGATE_START_AT = "aggregate-start-at";
    private static final String OPT_AGGREGATE_END_AT = "aggregate-end-at";
    private static final String OPT_NO_SUMMARY = "no-summary";
    private static final String GROUP_PLOT_SYNTHETIC = "group-plot-synthetic$";
    private static final String RATIO_PLOT_SYNTHETIC = "ratio-plot-synthetic$";
    private static final SimpleDateFormat[] sDateFormats = new SimpleDateFormat[]{new SimpleDateFormat("MM/dd/yyyy HH:mm:ss"), new SimpleDateFormat("MM/dd/yyyy HH:mm")};
    private static final String SUMMARY_CSV = "summary.csv";
    private File[] mConfs;
    private File[] mSrcDirs;
    private File mDestDir;
    private String mTitle;
    private Date mStartAt = null;
    private Date mEndAt = null;
    private Date mAggregateStartAt = null;
    private Date mAggregateEndAt = null;
    private boolean mSkipSummary;
    private List<ChartSettings> mSyntheticChartSettings = new ArrayList<ChartSettings>();
    private List<JFreeChart> mCharts = new ArrayList<JFreeChart>();
    private Set<DataColumn> mUniqueDataColumns;
    private Set<DataColumn> mUniqueStringColumns;
    private Map<String, Set<Pair<String, DataSeries>>> mColumnsByInfile;
    private Map<ChartSettings, JFreeChart> mChartMap = new HashMap<ChartSettings, JFreeChart>();
    private Map<DataColumn, DataSeries> mDataSeries;
    private Map<DataColumn, StringSeries> mStringSeries;
    private Map<DataColumn, Double> mAggregates;
    private Map<String, Double> mStats;
    private Aggregator mAggregator;
    private long mMinDate = Long.MAX_VALUE;
    private long mMaxDate = Long.MIN_VALUE;

    private static Options getOptions() {
        Options opts = new Options();
        opts.addOption("h", OPT_HELP, false, "prints this usage screen");
        Option confOption = new Option("c", OPT_CONF, true, "chart configuration xml files");
        confOption.setArgs(-2);
        confOption.setRequired(true);
        opts.addOption(confOption);
        Option srcDirOption = new Option("s", OPT_SRCDIR, true, "one or more directories where the csv files are located");
        srcDirOption.setArgs(-2);
        opts.addOption(srcDirOption);
        Option destDirOption = new Option("d", OPT_DESTDIR, true, "directory where the generated chart files are saved");
        opts.addOption(destDirOption);
        opts.addOption(null, OPT_TITLE, true, "chart title; defaults to last directory name of --srcdir value");
        opts.addOption(null, OPT_START_AT, true, "if specified, ignore all samples before this timestamp (MM/dd/yyyy HH:mm:ss)");
        opts.addOption(null, OPT_END_AT, true, "if specified, ignore all samples after this timestamp (MM/dd/yyyy HH:mm:ss)");
        opts.addOption(null, OPT_AGGREGATE_START_AT, true, "if specified, aggregate computation starts at this timestamp (MM/dd/yyyy HH:mm:ss)");
        opts.addOption(null, OPT_AGGREGATE_END_AT, true, "if specified, aggregate computation ends at this timestamp (MM/dd/yyyy HH:mm:ss)");
        opts.addOption(null, OPT_NO_SUMMARY, false, "skip summary data generation");
        return opts;
    }

    private static void usage(Options opts) {
        ChartUtil.usage(opts, null);
    }

    private static void usage(Options opts, String msg) {
        if (msg != null) {
            System.err.println(msg);
        }
        String invocation = "Usage: zmstat-chart -c <arg> -s <arg> -d <arg> [options]";
        PrintWriter pw = new PrintWriter(System.err, true);
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(pw, formatter.getWidth(), invocation, null, opts, formatter.getLeftPadding(), formatter.getDescPadding(), null);
        pw.flush();
        System.exit(1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Date parseTimestampOption(CommandLine cl, Options opts, String opt) {
        Date date = null;
        String str = cl.getOptionValue(opt);
        if (str != null) {
            for (int i = 0; i < sDateFormats.length && date == null; ++i) {
                try {
                    SimpleDateFormat simpleDateFormat = sDateFormats[i];
                    synchronized (simpleDateFormat) {
                        date = sDateFormats[i].parse(str);
                        continue;
                    }
                }
                catch (ParseException e) {
                    // empty catch block
                }
            }
            if (date == null) {
                ChartUtil.usage(opts, "Invalid --" + opt + "value \"" + str + "\"");
            }
        }
        return date;
    }

    public static void main(String[] args) throws Exception {
        GnuParser clParser = new GnuParser();
        Options opts = ChartUtil.getOptions();
        try {
            String title;
            boolean created;
            File destDir;
            String[] confs;
            CommandLine cl = clParser.parse(opts, args);
            if (cl.hasOption('h')) {
                ChartUtil.usage(opts);
            }
            if (!cl.hasOption('s') && !cl.hasOption('d')) {
                ChartUtil.usage(opts, "-s and -d options are required");
            }
            if (!cl.hasOption('s')) {
                ChartUtil.usage(opts, "Missing required -s option");
            }
            if (!cl.hasOption('d')) {
                ChartUtil.usage(opts, "Missing required -d option");
            }
            if ((confs = cl.getOptionValues(OPT_CONF)) == null || confs.length == 0) {
                ChartUtil.usage(opts, "Missing --conf option");
            }
            File[] confFiles = new File[confs.length];
            for (int i = 0; i < confs.length; ++i) {
                File conf = new File(confs[i]);
                if (!conf.exists()) {
                    System.err.printf("Configuration file %s does not exist\n", conf.getAbsolutePath());
                    System.exit(1);
                }
                confFiles[i] = conf;
            }
            String[] srcDirStrs = cl.getOptionValues(OPT_SRCDIR);
            if (srcDirStrs == null || srcDirStrs.length == 0) {
                ChartUtil.usage(opts, "Missing --srcdir option");
            }
            ArrayList<File> srcDirsList = new ArrayList<File>(srcDirStrs.length);
            for (int i = 0; i < srcDirStrs.length; ++i) {
                File srcDir = new File(srcDirStrs[i]);
                if (srcDir.exists()) {
                    srcDirsList.add(srcDir);
                    continue;
                }
                System.err.printf("Source directory %s does not exist\n", srcDir.getAbsolutePath());
            }
            if (srcDirsList.size() < 1) {
                ChartUtil.usage(opts, "No valid source directory found");
            }
            File[] srcDirs = new File[srcDirsList.size()];
            srcDirsList.toArray(srcDirs);
            String destDirStr = cl.getOptionValue(OPT_DESTDIR);
            if (destDirStr == null) {
                ChartUtil.usage(opts, "Missing --destdir option");
            }
            if (!(destDir = new File(destDirStr)).exists() && !(created = destDir.mkdirs())) {
                System.err.printf("Unable to create destination directory %s\n", destDir.getAbsolutePath());
                System.exit(1);
            }
            if (!destDir.canWrite()) {
                System.err.printf("Destination directory %s is not writable\n", destDir.getAbsolutePath());
                System.exit(1);
            }
            if ((title = cl.getOptionValue(OPT_TITLE)) == null) {
                title = srcDirs[0].getAbsoluteFile().getName();
            }
            Date startAt = ChartUtil.parseTimestampOption(cl, opts, OPT_START_AT);
            Date endAt = ChartUtil.parseTimestampOption(cl, opts, OPT_END_AT);
            Date aggStartAt = ChartUtil.parseTimestampOption(cl, opts, OPT_AGGREGATE_START_AT);
            Date aggEndAt = ChartUtil.parseTimestampOption(cl, opts, OPT_AGGREGATE_END_AT);
            boolean noSummary = cl.hasOption('n');
            ChartUtil app = new ChartUtil(confFiles, srcDirs, destDir, title, startAt, endAt, aggStartAt, aggEndAt, noSummary);
            app.doit();
        }
        catch (Exception e) {
            e.printStackTrace();
            System.err.println();
            ChartUtil.usage(opts);
        }
    }

    public ChartUtil(File[] confFiles, File[] srcDirs, File destDir, String title, Date startAt, Date endAt, Date aggregateStartAt, Date aggregateEndAt, boolean skipSummary) {
        this.mConfs = confFiles;
        this.mSrcDirs = srcDirs;
        this.mDestDir = destDir;
        this.mTitle = title;
        this.mStartAt = startAt != null ? startAt : new Date(Long.MIN_VALUE);
        this.mEndAt = endAt != null ? endAt : new Date(Long.MAX_VALUE);
        this.mAggregateStartAt = aggregateStartAt != null ? aggregateStartAt : new Date(Long.MIN_VALUE);
        this.mAggregateEndAt = aggregateEndAt != null ? aggregateEndAt : new Date(Long.MAX_VALUE);
        this.mSkipSummary = skipSummary;
        this.mUniqueDataColumns = new HashSet<DataColumn>();
        this.mUniqueStringColumns = new HashSet<DataColumn>();
        this.mColumnsByInfile = new HashMap<String, Set<Pair<String, DataSeries>>>();
        this.mStringSeries = new HashMap<DataColumn, StringSeries>();
        this.mDataSeries = new HashMap<DataColumn, DataSeries>();
        this.mAggregates = new HashMap<DataColumn, Double>();
        this.mAggregator = new Aggregator();
        this.mStats = new HashMap<String, Double>();
    }

    private void doit() throws Exception {
        List<ChartSettings> allSettings = this.getAllChartSettings(this.mConfs);
        this.readCsvFiles();
        ArrayList<ChartSettings> outDocSettings = new ArrayList<ChartSettings>();
        HashSet<String> outDocNames = new HashSet<String>();
        Iterator<ChartSettings> i = allSettings.iterator();
        while (i.hasNext()) {
            ChartSettings cs = i.next();
            this.mCharts.addAll(this.createJFReeChart(cs));
            if (cs.getOutDocument() == null || cs.getGroupPlots().size() == 0) {
                this.computeAggregates(cs, this.mAggregateStartAt, this.mAggregateEndAt);
                continue;
            }
            if (cs.getOutDocument() == null) continue;
            outDocSettings.add(cs);
            outDocNames.add(cs.getOutDocument());
            i.remove();
        }
        for (ChartSettings cs : this.mSyntheticChartSettings) {
            this.computeAggregates(cs, this.mAggregateStartAt, this.mAggregateEndAt);
            outDocNames.add(cs.getOutDocument());
        }
        outDocSettings.addAll(this.mSyntheticChartSettings);
        outDocNames.remove(null);
        this.lineUpAxes();
        this.writeAllCharts(allSettings, outDocNames);
        this.writeOutDocCharts(this.mSyntheticChartSettings, outDocNames);
        if (!this.mSkipSummary) {
            this.writeSummary(allSettings);
        }
    }

    private List<ChartSettings> getAllChartSettings(File[] chartDefs) throws Exception {
        ArrayList<ChartSettings> allSettings = new ArrayList<ChartSettings>();
        for (File def : chartDefs) {
            List<ChartSettings> settings = XMLChartConfig.load(def);
            if (settings.size() == 0) {
                System.err.println("No chart settings found in " + def.getAbsolutePath());
                System.exit(1);
            }
            for (ChartSettings cs : settings) {
                DataColumn dc;
                String column;
                String infile;
                ArrayList<PlotSettings> plots = new ArrayList<PlotSettings>();
                plots.addAll(cs.getPlots());
                plots.addAll(cs.getGroupPlots());
                for (PlotSettings ps : plots) {
                    infile = ps.getInfile();
                    column = ps.getDataColumn();
                    if (column == null) {
                        String[] top = ps.getRatioTop().split("\\+");
                        String[] bottom = ps.getRatioBottom().split("\\+");
                        ArrayList<String> cols = new ArrayList<String>();
                        cols.addAll(Arrays.asList(top));
                        cols.addAll(Arrays.asList(bottom));
                        for (String c : cols) {
                            DataColumn dc2 = new DataColumn(infile, c);
                            this.mUniqueDataColumns.add(dc2);
                        }
                        continue;
                    }
                    dc = new DataColumn(infile, column);
                    this.mUniqueDataColumns.add(dc);
                }
                for (GroupPlotSettings gps : cs.getGroupPlots()) {
                    infile = gps.getInfile();
                    column = gps.getGroupBy();
                    dc = new DataColumn(infile, column);
                    this.mUniqueStringColumns.add(dc);
                    this.mStringSeries.put(dc, new StringSeries());
                }
            }
            allSettings.addAll(settings);
        }
        this.validateChartSettings(allSettings);
        for (DataColumn dc : this.mUniqueDataColumns) {
            String infile = dc.getInfile();
            String column = dc.getColumn();
            Set<Pair<String, DataSeries>> cols = this.mColumnsByInfile.get(infile);
            if (cols == null) {
                cols = new HashSet<Pair<String, DataSeries>>();
                this.mColumnsByInfile.put(infile, cols);
            }
            DataSeries series = new DataSeries();
            this.mDataSeries.put(dc, series);
            Pair<String, DataSeries> colSeries = new Pair<String, DataSeries>(column, series);
            cols.add(colSeries);
        }
        return allSettings;
    }

    private void computeAggregates(ChartSettings cs, Date startAt, Date endAt) {
        double agg = 0.0;
        List<PlotSettings> plots = cs.getPlots();
        for (PlotSettings ps : plots) {
            DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
            String key = ps.getInfile() + ":" + ps.getDataColumn() + ":" + ps.getAggregateFunction();
            PlotDataIterator pdIter = new PlotDataIterator(ps, this.mDataSeries.get(dc));
            agg = this.mAggregator.compute(pdIter, ps.getAggregateFunction(), startAt, endAt, key);
            this.mAggregates.put(dc, agg);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeSummary(List<ChartSettings> allSettings) throws IOException {
        File resultCsv = new File(this.mDestDir, SUMMARY_CSV);
        FileWriter writer = null;
        try {
            writer = new FileWriter(resultCsv);
            int count = 0;
            String key = "";
            Double val = null;
            for (ChartSettings cs : allSettings) {
                for (PlotSettings ps : cs.getPlots()) {
                    DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
                    key = ps.getInfile() + ":" + ps.getDataColumn() + ":" + ps.getAggregateFunction();
                    val = this.mAggregates.get(dc);
                    ++count;
                    this.mStats.put(key, val);
                }
            }
            Iterator<String> keyset = this.mStats.keySet().iterator();
            StringBuilder keys = new StringBuilder();
            StringBuilder vals = new StringBuilder();
            while (keyset.hasNext()) {
                key = keyset.next();
                keys.append(key).append(",");
                vals.append(ChartUtil.formatDouble(this.mStats.get(key))).append(",");
            }
            writer.write(keys.toString());
            writer.write("\n");
            writer.write(vals.toString());
            writer.write("\n");
        }
        finally {
            if (writer != null) {
                writer.close();
            }
        }
        SummaryAnalyzer analyzer = new SummaryAnalyzer(allSettings);
        analyzer.writeReport(resultCsv);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeOutDocCharts(List<ChartSettings> outChartSettings, Set<String> linkedDocuments) throws IOException {
        FileWriter writer;
        File file;
        for (String filename : linkedDocuments) {
            file = new File(this.mDestDir, filename);
            writer = new FileWriter(file, false);
            try {
                writer.write("<html><head><title>");
                writer.write(StringUtil.escapeHtml(this.mTitle) + ": " + filename);
                writer.write("</title><body bgcolor=\"#eeeeee\"><h1>");
                writer.write(StringUtil.escapeHtml(this.mTitle) + ": " + filename);
                writer.write("</h1>\n");
            }
            finally {
                writer.close();
            }
        }
        for (ChartSettings cs : outChartSettings) {
            JFreeChart chart = this.mChartMap.get(cs);
            if (chart == null || !this.hasData(chart)) continue;
            this.writeChart(chart, cs);
            writer = new FileWriter(new File(this.mDestDir, cs.getOutDocument()), true);
            try {
                List<PlotSettings> plots = cs.getPlots();
                String statString = "";
                boolean first = true;
                for (PlotSettings ps : plots) {
                    DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
                    DataSeries ds = this.mDataSeries.get(dc);
                    if (ds == null || ds.size() == 0) continue;
                    if (first) {
                        first = false;
                    } else {
                        statString = statString + " &nbsp;&nbsp; ";
                    }
                    statString = statString + ps.getAggregateFunction() + "(" + ps.getLegend() + ") = " + ChartUtil.formatDouble(this.mAggregates.get(dc));
                }
                writer.write("<a name=\"" + cs.getOutfile() + "\">");
                writer.write("<h3>" + cs.getTitle() + "</h3></a>\n");
                writer.write("<h5>" + statString + "</h5>\n");
                writer.write(String.format("<img src=\"%s\" width=\"%d\" height=\"%d\">\n", cs.getOutfile(), cs.getWidth(), cs.getHeight()));
            }
            finally {
                writer.close();
            }
        }
        for (String filename : linkedDocuments) {
            file = new File(this.mDestDir, filename);
            writer = new FileWriter(file, true);
            try {
                writer.write("</body></html>\n");
            }
            finally {
                writer.close();
            }
        }
    }

    private void writeAllCharts(List<ChartSettings> allSettings, Set<String> linkedDocuments) throws IOException {
        for (ChartSettings cs : allSettings) {
            JFreeChart chart = this.mChartMap.get(cs);
            if (chart == null) continue;
            this.writeChart(chart, cs);
        }
        this.writeIndexHtml(allSettings, linkedDocuments);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeIndexHtml(List<ChartSettings> allSettings, Set<String> linkedDocuments) throws IOException {
        System.out.println("Writing index.html");
        OutputStreamWriter writer = null;
        try {
            writer = new FileWriter(new File(this.mDestDir, "index.html"));
            writer.write("<html>\n<head>\n<title>" + StringUtil.escapeHtml(this.mTitle) + " " + DateFormat.getDateInstance().format(this.mMinDate) + "</title>\n</head>\n<body bgcolor=\"#eeeeee\">\n");
            writer.write("<h1>" + StringUtil.escapeHtml(this.mTitle) + "</h1>\n");
            if (linkedDocuments.size() > 0) {
                writer.write("<h2>Additional charts</h2>");
                writer.write("<ul>");
                for (String document : linkedDocuments) {
                    writer.write("<li><a href=\"" + document + "\">");
                    writer.write(document);
                    writer.write("</a></li>\n");
                }
                writer.write("</ul>\n");
            }
            ArrayList<String> noData = new ArrayList<String>();
            int count = 0;
            for (ChartSettings cs : allSettings) {
                JFreeChart chart = this.mChartMap.get(cs);
                if (chart == null) continue;
                List<PlotSettings> plots = cs.getPlots();
                String statString = "";
                boolean first = true;
                for (PlotSettings ps : plots) {
                    DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
                    DataSeries ds = this.mDataSeries.get(dc);
                    if (ds == null || ds.size() == 0) continue;
                    if (first) {
                        first = false;
                    } else {
                        statString = statString + " &nbsp;&nbsp; ";
                    }
                    statString = statString + ps.getAggregateFunction() + "(" + ps.getLegend() + ") = " + ChartUtil.formatDouble(this.mAggregates.get(dc));
                    ++count;
                }
                if (this.hasData(chart)) {
                    writer.write("<a name=\"" + cs.getOutfile() + "\">");
                    writer.write("<h3>" + cs.getTitle() + "</h3></a>\n");
                    writer.write("<h5>" + statString + "</h5>\n");
                    writer.write(String.format("<img src=\"%s\" width=\"%d\" height=\"%d\">\n", cs.getOutfile(), cs.getWidth(), cs.getHeight()));
                    continue;
                }
                noData.add(cs.getTitle());
            }
            if (noData.size() > 0) {
                writer.write("<h3>No data available for the following charts:</h3>\n");
                writer.write("<p>\n");
                for (String str : noData) {
                    writer.write(StringUtil.escapeHtml(str));
                    writer.write("\n");
                }
                writer.write("<p>\n");
            }
            if (!this.mSkipSummary) {
                writer.write("<h4><a href=\"summary.txt\">Summary</a></h4>\n");
            }
            writer.write("</body>\n</html>\n");
        }
        finally {
            if (writer != null) {
                writer.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Date readTS(CsvReader r, String ctx) throws ParseException {
        Date ts = null;
        String tstamp = r.getValue("timestamp");
        if (tstamp == null) {
            throw new ParseException(ctx + ": no timestamp found.", 0);
        }
        ParseException lastException = null;
        for (int i = 0; i < sDateFormats.length && ts == null; ++i) {
            try {
                SimpleDateFormat simpleDateFormat = sDateFormats[i];
                synchronized (simpleDateFormat) {
                    ts = sDateFormats[i].parse(tstamp);
                }
            }
            catch (ParseException e) {
                lastException = e;
            }
            if (lastException == null) continue;
            throw lastException;
        }
        return ts;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readCsvFiles() throws Exception {
        Date ts;
        int line;
        CsvReader csv;
        String inFilename;
        Date minDate = null;
        Date maxDate = null;
        for (DataColumn c : this.mUniqueStringColumns) {
            inFilename = c.getInfile();
            MultipleDirsFileReader reader = null;
            StringSeries data = this.mStringSeries.get(c);
            try {
                reader = new MultipleDirsFileReader(inFilename, this.mSrcDirs);
            }
            catch (FileNotFoundException e) {
                System.out.printf("CSV file %s not found; Skipping...\n", inFilename);
                continue;
            }
            csv = null;
            try {
                csv = new CsvReader(reader);
                line = 1;
                while (csv.hasNext()) {
                    String ctx = inFilename + ", line " + ++line;
                    ts = null;
                    try {
                        ts = this.readTS(csv, ctx);
                    }
                    catch (ParseException e) {
                        System.out.println(ctx + ": " + e);
                        continue;
                    }
                    if (ts.before(this.mStartAt) || ts.after(this.mEndAt)) continue;
                    if (minDate == null) {
                        minDate = ts;
                        maxDate = ts;
                    } else {
                        if (ts.compareTo(minDate) < 0) {
                            minDate = ts;
                        }
                        if (ts.compareTo(maxDate) > 0) {
                            maxDate = ts;
                        }
                    }
                    String value = csv.getValue(c.getColumn());
                    data.AddEntry(ts, value);
                }
            }
            finally {
                if (csv == null) continue;
                csv.close();
            }
        }
        for (Map.Entry<String, Set<Pair<String, DataSeries>>> entry : this.mColumnsByInfile.entrySet()) {
            inFilename = entry.getKey();
            Set<Pair<String, DataSeries>> columns = entry.getValue();
            System.out.println("Reading CSV " + inFilename);
            MultipleDirsFileReader reader = null;
            try {
                reader = new MultipleDirsFileReader(inFilename, this.mSrcDirs);
            }
            catch (FileNotFoundException e) {
                System.out.printf("CSV file %s not found; Skipping...\n", inFilename);
                continue;
            }
            csv = null;
            try {
                csv = new CsvReader(reader);
                line = 1;
                while (csv.hasNext()) {
                    String context = inFilename + ", line " + ++line;
                    ts = null;
                    try {
                        ts = this.readTS(csv, context);
                    }
                    catch (ParseException e) {
                        System.out.println(context + ": " + e);
                        continue;
                    }
                    if (ts.before(this.mStartAt) || ts.after(this.mEndAt)) continue;
                    if (minDate == null) {
                        minDate = ts;
                        maxDate = ts;
                    } else {
                        if (ts.compareTo(minDate) < 0) {
                            minDate = ts;
                        }
                        if (ts.compareTo(maxDate) > 0) {
                            maxDate = ts;
                        }
                    }
                    for (Pair<String, DataSeries> colSeries : columns) {
                        String column = colSeries.getFirst();
                        DataSeries series = colSeries.getSecond();
                        String val = csv.getValue(column);
                        if (!StringUtil.isNullOrEmpty(val)) {
                            try {
                                double d = Double.parseDouble(val);
                                try {
                                    series.AddEntry(ts, d);
                                }
                                catch (SeriesException e) {
                                    System.out.printf("Can't add sample to series: timestamp=%s, value=%s\n", ts, val);
                                    e.printStackTrace(System.out);
                                }
                            }
                            catch (NumberFormatException e) {
                                System.out.println(String.format("%s: unable to parse value '%s' for %s: %s", context, val, column, e));
                            }
                            continue;
                        }
                        series.AddEntry(ts, 0.0);
                    }
                }
                for (Pair<String, DataSeries> colSeries : columns) {
                    String column = colSeries.getFirst();
                    DataSeries series = colSeries.getSecond();
                    System.out.format("Adding %d %s points between %s and %s.\n\n", series.size(), column, minDate, maxDate);
                }
            }
            finally {
                if (csv == null) continue;
                csv.close();
            }
        }
        this.adustSampleRange(minDate, maxDate);
    }

    private void adustSampleRange(Date minDate, Date maxDate) {
        long chartMaxDate;
        long chartMinDate;
        if (minDate != null && (chartMinDate = minDate.getTime()) < this.mMinDate) {
            this.mMinDate = chartMinDate;
        }
        if (maxDate != null && (chartMaxDate = maxDate.getTime()) > this.mMaxDate) {
            this.mMaxDate = chartMaxDate;
        }
    }

    private void validateChartSettings(List<ChartSettings> allSettings) throws ParseException {
        HashSet<String> usedFilenames = new HashSet<String>();
        for (ChartSettings cs : allSettings) {
            String filename = cs.getOutfile();
            if (usedFilenames.contains(filename)) {
                throw new ParseException("Found two charts that write " + filename, 0);
            }
            usedFilenames.add(filename);
        }
    }

    /*
     * WARNING - void declaration
     */
    private List<JFreeChart> createJFReeChart(ChartSettings cs) {
        DataSeries ds;
        double minValue = Double.MAX_VALUE;
        double maxValue = Double.MIN_VALUE;
        double d = 0.0;
        double count = 0.0;
        double total = 0.0;
        TimeSeriesCollection data = new TimeSeriesCollection();
        ArrayList<ChartSettings> syntheticSettings = new ArrayList<ChartSettings>();
        for (GroupPlotSettings gps : cs.getGroupPlots()) {
            List indices;
            String groupBy = gps.getGroupBy();
            DataColumn dc = new DataColumn(gps.getInfile(), groupBy);
            StringSeries groupBySeries = this.mStringSeries.get(dc);
            dc = new DataColumn(gps.getInfile(), gps.getDataColumn());
            ds = this.mDataSeries.get(dc);
            int idx = 0;
            HashMap<String, ArrayList<Integer>> groups = new HashMap<String, ArrayList<Integer>>();
            for (StringEntry stringEntry : groupBySeries.dataCollection) {
                String g = stringEntry.getVal();
                indices = (ArrayList<Integer>)groups.get(g);
                if (indices == null) {
                    indices = new ArrayList<Integer>();
                    groups.put(g, (ArrayList<Integer>)indices);
                }
                indices.add(idx);
                ++idx;
            }
            for (Map.Entry entry : groups.entrySet()) {
                String groupByValue = (String)entry.getKey();
                if (gps.getIgnoreSet().contains(groupByValue)) continue;
                indices = (List)entry.getValue();
                DataSeries syntheticDS = new DataSeries();
                DataColumn c = new DataColumn(gps.getInfile(), GROUP_PLOT_SYNTHETIC + groupByValue + ":" + gps.getDataColumn());
                Iterator i$ = indices.iterator();
                while (i$.hasNext()) {
                    int i = (Integer)i$.next();
                    Entry e = ds.get(i);
                    syntheticDS.AddEntry(e.getTimestamp(), e.getVal());
                }
                this.mDataSeries.put(c, syntheticDS);
                PlotSettings syntheticPlot = new PlotSettings(groupByValue, c.getInfile(), c.getColumn(), gps.getShowRaw(), gps.getShowMovingAvg(), gps.getMovingAvgPoints(), gps.getMultiplier(), gps.getDivisor(), gps.getNonNegative(), gps.getPercentTime(), gps.getDataFunction(), gps.getAggregateFunction(), gps.getOptional(), null, null);
                cs.addPlot(syntheticPlot);
                if (cs.getOutDocument() == null) continue;
                ChartSettings s = new ChartSettings(String.format(cs.getTitle(), groupByValue), cs.getCategory(), String.format(cs.getOutfile(), groupByValue), cs.getXAxis(), cs.getYAxis(), cs.getAllowLogScale(), cs.getPlotZero(), cs.getWidth(), cs.getHeight(), null, cs.getTopPlots(), cs.getTopPlotsType());
                s.addPlot(syntheticPlot);
                syntheticSettings.add(s);
            }
        }
        if (cs.getOutDocument() != null && cs.getGroupPlots().size() != 0) {
            ArrayList<JFreeChart> charts = new ArrayList<JFreeChart>();
            for (ChartSettings c : syntheticSettings) {
                charts.addAll(this.createJFReeChart(c));
                c.setOutDocument(cs.getOutDocument());
            }
            this.mSyntheticChartSettings.addAll(syntheticSettings);
            return charts;
        }
        List<PlotSettings> plots = cs.getPlots();
        if (cs.getTopPlots() > 0 && plots.size() > cs.getTopPlots()) {
            String aggregateFunction = cs.getTopPlotsType().name().toLowerCase();
            System.out.println(String.format("Reducing %d to %d plots for chart '%s'", plots.size(), cs.getTopPlots(), cs.getTitle()));
            ArrayList<PlotAggregatePair> aggregates = new ArrayList<PlotAggregatePair>();
            for (PlotSettings ps : plots) {
                DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
                String key = ps.getInfile() + ":" + ps.getDataColumn() + ":" + ps.getAggregateFunction();
                PlotDataIterator pdIter = new PlotDataIterator(ps, this.mDataSeries.get(dc));
                double aggregate = this.mAggregator.compute(pdIter, aggregateFunction, this.mAggregateStartAt, this.mAggregateEndAt, key);
                aggregates.add(new PlotAggregatePair(ps, aggregate));
            }
            Collections.sort(aggregates);
            while (aggregates.size() > cs.getTopPlots()) {
                PlotAggregatePair pair = (PlotAggregatePair)aggregates.remove(0);
                plots.remove(pair.ps);
            }
        }
        for (PlotSettings ps : plots) {
            String columnName = ps.getDataColumn();
            if (columnName == null) {
                int i;
                void var23_36;
                void var23_34;
                columnName = RATIO_PLOT_SYNTHETIC + ps.getRatioTop() + "/" + ps.getRatioBottom();
                String infile = ps.getInfile();
                String[] top = ps.getRatioTop().split("\\+");
                String[] bottom = ps.getRatioBottom().split("\\+");
                DataColumn[] ratioTop = new DataColumn[top.length];
                DataColumn[] ratioBottom = new DataColumn[bottom.length];
                boolean bl = false;
                int j = top.length;
                while (var23_34 < j) {
                    ratioTop[var23_34] = new DataColumn(infile, top[var23_34]);
                    ++var23_34;
                }
                boolean bl2 = false;
                j = bottom.length;
                while (var23_36 < j) {
                    ratioBottom[var23_36] = new DataColumn(infile, bottom[var23_36]);
                    ++var23_36;
                }
                DataSeries[] dataSeriesArray = new DataSeries[ratioTop.length];
                DataSeries[] bottomData = new DataSeries[ratioBottom.length];
                int j2 = ratioTop.length;
                for (i = 0; i < j2; ++i) {
                    dataSeriesArray[i] = this.mDataSeries.get(ratioTop[i]);
                }
                int j3 = ratioBottom.length;
                for (i = 0; i < j3; ++i) {
                    bottomData[i] = this.mDataSeries.get(ratioBottom[i]);
                }
                DataSeries ds2 = new DataSeries();
                int j32 = dataSeriesArray[0].size();
                for (int i3 = 0; i3 < j32; ++i3) {
                    Entry e;
                    int m;
                    double topValue = 0.0;
                    double bottomValue = 0.0;
                    double ratio = 0.0;
                    Entry lastEntry = null;
                    int n = dataSeriesArray.length;
                    for (m = 0; m < n; ++m) {
                        e = dataSeriesArray[m].get(i3);
                        topValue += e.getVal();
                    }
                    n = bottomData.length;
                    for (m = 0; m < n; ++m) {
                        e = bottomData[m].get(i3);
                        bottomValue += e.getVal();
                        lastEntry = e;
                    }
                    if (bottomValue != 0.0) {
                        ratio = topValue / bottomValue;
                    }
                    assert (lastEntry != null);
                    ds2.AddEntry(lastEntry.getTimestamp(), ratio);
                }
                this.mDataSeries.put(new DataColumn(infile, columnName), ds2);
                ps.setDataColumn(columnName);
            }
            DataColumn dc = new DataColumn(ps.getInfile(), ps.getDataColumn());
            ds = this.mDataSeries.get(dc);
            TimeSeries ts = new TimeSeries(ps.getLegend(), FixedMillisecond.class);
            int numSamples = 0;
            PlotDataIterator pdIter = new PlotDataIterator(ps, ds);
            while (pdIter.hasNext()) {
                Object object = pdIter.next();
                Date tstamp = (Date)((Pair)object).getFirst();
                double val = (Double)((Pair)object).getSecond();
                if (val != 0.0 || cs.getPlotZero()) {
                    if (d < minValue) {
                        minValue = val;
                    }
                    if (d > maxValue) {
                        maxValue = val;
                    }
                    count += 1.0;
                    total += val;
                    try {
                        ts.addOrUpdate((RegularTimePeriod)new FixedMillisecond(tstamp), val);
                    }
                    catch (SeriesException e) {
                        e.printStackTrace(System.out);
                    }
                }
                ++numSamples;
            }
            if (numSamples == 0 && ps.getOptional()) {
                System.out.format("Skipping optional plot %s (no data sample found)\n\n", ps.getLegend());
                continue;
            }
            System.out.format("Adding %d %s points to %s.\n\n", ds.size(), ps.getLegend(), cs.getOutfile());
            if (ps.getShowRaw()) {
                data.addSeries(ts);
            }
            if (!ps.getShowMovingAvg()) continue;
            int numPoints = ps.getMovingAvgPoints();
            if (numPoints == -1) {
                numPoints = ts.getItemCount() / 200;
            }
            if (numPoints >= 2) {
                TimeSeries timeSeries = MovingAverage.createPointMovingAverage((TimeSeries)ts, (String)(ps.getLegend() + " (moving avg)"), (int)numPoints);
                data.addSeries(timeSeries);
                continue;
            }
            System.out.println("Not enough data to display moving average for " + ps.getLegend());
            data.addSeries(ts);
        }
        boolean legend = data.getSeriesCount() > 1;
        JFreeChart chart = ChartFactory.createTimeSeriesChart(null, (String)cs.getXAxis(), (String)cs.getYAxis(), (XYDataset)data, (boolean)legend, (boolean)false, (boolean)false);
        if (cs.getAllowLogScale() && minValue > 0.0 && maxValue > 0.0 && maxValue > 20.0 * (total / count) && maxValue / minValue > 100.0) {
            XYPlot plot = (XYPlot)chart.getPlot();
            ValueAxis oldAxis = plot.getRangeAxis();
            LogarithmicAxis newAxis = new LogarithmicAxis(oldAxis.getLabel());
            plot.setRangeAxis((ValueAxis)newAxis);
        }
        this.mChartMap.put(cs, chart);
        return Arrays.asList(chart);
    }

    private boolean hasData(JFreeChart chart) {
        if (chart == null) {
            return false;
        }
        XYPlot plot = chart.getXYPlot();
        XYDataset data = plot.getDataset();
        int numPoints = 0;
        for (int i = 0; i < data.getSeriesCount(); ++i) {
            numPoints += data.getItemCount(i);
        }
        return numPoints != 0;
    }

    private void lineUpAxes() {
        for (JFreeChart chart : this.mCharts) {
            XYPlot plot = (XYPlot)chart.getPlot();
            DateAxis axis = (DateAxis)plot.getDomainAxis();
            Date chartMinDate = axis.getMinimumDate();
            Date chartMaxDate = axis.getMaximumDate();
            if (chartMinDate != null && this.mMinDate < chartMinDate.getTime()) {
                axis.setMinimumDate(new Date(this.mMinDate));
            }
            if (chartMaxDate == null || this.mMaxDate <= chartMaxDate.getTime()) continue;
            axis.setMaximumDate(new Date(this.mMaxDate));
        }
    }

    private void writeChart(JFreeChart chart, ChartSettings cs) throws IOException {
        String filename = cs.getOutfile();
        System.out.println("Writing " + filename);
        File file = new File(this.mDestDir, filename);
        if (cs.getImageType() == ChartSettings.ImageType.PNG) {
            ChartUtilities.saveChartAsPNG((File)file, (JFreeChart)chart, (int)cs.getWidth(), (int)cs.getHeight());
        } else {
            ChartUtilities.saveChartAsJPEG((File)file, (float)90.0f, (JFreeChart)chart, (int)cs.getWidth(), (int)cs.getHeight());
        }
    }

    private static String formatDouble(double d) {
        DecimalFormat formatter = new DecimalFormat("0.00");
        return formatter.format(d);
    }

    class Aggregator {
        HashMap<String, Integer> set = new HashMap();

        private double getAvg(PlotDataIterator pdIter, Date startAt, Date endAt) {
            int count = 0;
            double val = 0.0;
            while (pdIter.hasNext()) {
                Object entry = pdIter.next();
                Date date = (Date)((Pair)entry).getFirst();
                if (date.before(startAt) || date.after(endAt)) continue;
                val += ((Double)((Pair)entry).getSecond()).doubleValue();
                ++count;
            }
            return count > 0 ? val / (double)count : 0.0;
        }

        private double getMax(PlotDataIterator pdIter, Date startAt, Date endAt) {
            double val = 0.0;
            while (pdIter.hasNext()) {
                Object entry = pdIter.next();
                Date date = (Date)((Pair)entry).getFirst();
                if (date.before(startAt) || date.after(endAt)) continue;
                val = Math.max(val, (Double)((Pair)entry).getSecond());
            }
            return val;
        }

        private double getMaxPercentage(PlotDataIterator pdIter, Date startAt, Date endAt, String key) {
            double val = 0.0;
            while (pdIter.hasNext()) {
                Object entry = pdIter.next();
                Date date = (Date)((Pair)entry).getFirst();
                if (date.before(startAt) || date.after(endAt)) continue;
                val = (Double)((Pair)entry).getSecond();
            }
            val = 0.0;
            return val;
        }

        private double getLast(PlotDataIterator pdIter, Date startAt, Date endAt) {
            double val = 0.0;
            while (pdIter.hasNext()) {
                Object entry = pdIter.next();
                Date date = (Date)((Pair)entry).getFirst();
                if (date.before(startAt) || date.after(endAt)) continue;
                val = (Double)((Pair)entry).getSecond();
            }
            return val;
        }

        public double compute(PlotDataIterator pdIter, String aggFunc, Date startAt, Date endAt, String key) {
            if (aggFunc.equalsIgnoreCase("last")) {
                return this.getLast(pdIter, startAt, endAt);
            }
            if (aggFunc.equalsIgnoreCase("max")) {
                return this.getMax(pdIter, startAt, endAt);
            }
            if (aggFunc.equalsIgnoreCase("maxPercentage")) {
                return this.getMaxPercentage(pdIter, startAt, endAt, key);
            }
            return this.getAvg(pdIter, startAt, endAt);
        }
    }

    class StringSeries {
        List<StringEntry> dataCollection = new ArrayList<StringEntry>();

        StringSeries() {
        }

        public void AddEntry(Date t, String s) {
            StringEntry entry = new StringEntry(t, s);
            this.dataCollection.add(entry);
        }

        public int size() {
            return this.dataCollection.size();
        }

        public StringEntry get(int index) {
            return this.dataCollection.get(index);
        }
    }

    class StringEntry {
        Date timestamp;
        String value;

        public StringEntry(Date t, String v) {
            this.timestamp = t;
            this.value = v;
        }

        public Date getTimestamp() {
            return this.timestamp;
        }

        public String getVal() {
            return this.value;
        }
    }

    class DataSeries {
        List<Entry> dataCollection = new ArrayList<Entry>();

        DataSeries() {
        }

        public void AddEntry(Date t, double d) {
            Entry entry = new Entry(t, d);
            this.dataCollection.add(entry);
        }

        public int size() {
            return this.dataCollection.size();
        }

        public Entry get(int index) {
            return this.dataCollection.get(index);
        }
    }

    class Entry {
        Date timestamp;
        double value;

        public Entry(Date t, double v) {
            this.timestamp = t;
            this.value = v;
        }

        public Date getTimestamp() {
            return this.timestamp;
        }

        public double getVal() {
            return this.value;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static class PlotAggregatePair
    implements Comparable<PlotAggregatePair> {
        private PlotSettings ps;
        private double aggregate;

        PlotAggregatePair(PlotSettings ps, double aggregate) {
            this.ps = ps;
            this.aggregate = aggregate;
        }

        @Override
        public int compareTo(PlotAggregatePair o) {
            double comparison = o.aggregate - this.aggregate;
            if (comparison == 0.0) {
                return 0;
            }
            return comparison > 0.0 ? -1 : 1;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static class PlotDataIterator
    implements Iterator<Pair<Date, Double>> {
        private PlotSettings mPlotSettings;
        private DataSeries mDataSeries;
        private long mLastTstamp;
        private Func mFunc;
        private double mSum;
        private double mLast;
        private int mIndex;

        public PlotDataIterator(PlotSettings ps, DataSeries ds) {
            this.mPlotSettings = ps;
            this.mDataSeries = ds;
            this.mIndex = 0;
            this.mSum = 0.0;
            this.mLast = Double.NaN;
            this.mLastTstamp = 0L;
            String func = this.mPlotSettings.getDataFunction();
            this.mFunc = "diff".equals(func) ? Func.DIFF : ("sum".equals(func) ? Func.SUM : Func.IDENTITY);
        }

        @Override
        public boolean hasNext() {
            return this.mIndex < this.mDataSeries.size();
        }

        @Override
        public Pair<Date, Double> next() {
            if (!this.hasNext()) {
                return null;
            }
            Entry entry = this.mDataSeries.get(this.mIndex);
            ++this.mIndex;
            double val = entry.getVal();
            val *= this.mPlotSettings.getMultiplier();
            double divisor = this.mPlotSettings.getDivisor();
            if (divisor != 0.0) {
                val /= divisor;
            }
            if (this.mFunc.equals((Object)Func.DIFF)) {
                if (Double.isNaN(this.mLast)) {
                    this.mLast = val;
                }
                double diff = val - this.mLast;
                this.mLast = val;
                val = diff;
            } else if (this.mFunc.equals((Object)Func.SUM)) {
                this.mSum += val;
                val = this.mSum;
            }
            if (val < 0.0 && this.mPlotSettings.getNonNegative()) {
                val = 0.0;
            }
            Date t = entry.getTimestamp();
            long tl = t.getTime();
            if (this.mPlotSettings.getPercentTime()) {
                long dt;
                val = this.mLastTstamp > 0L ? ((dt = tl - this.mLastTstamp) > 0L ? (val /= (double)dt) : 0.0) : 0.0;
                val *= 100.0;
            }
            this.mLastTstamp = tl;
            return new Pair<Date, Double>(t, val);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }

        /*
         * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
         */
        private static enum Func {
            IDENTITY,
            DIFF,
            SUM;

        }
    }

    private static class MultipleDirsFileReader
    extends Reader {
        private String mFilename;
        private File[] mDirs;
        private int mFileIndex;
        private Reader mReader;

        public MultipleDirsFileReader(String filename, File[] dirs) throws IOException {
            this.mFilename = filename;
            this.mDirs = dirs;
            this.mFileIndex = -1;
            this.openNextReader();
            if (this.mReader == null) {
                throw new FileNotFoundException("File " + filename + " not found in any of the source directories");
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void openNextReader() throws IOException {
            if (this.mReader != null) {
                this.mReader.close();
                this.mReader = null;
            }
            while (this.mFileIndex < this.mDirs.length - 1) {
                String filename;
                ++this.mFileIndex;
                File dir = this.mDirs[this.mFileIndex];
                File file = new File(dir, this.mFilename);
                if (!file.exists()) {
                    if (!this.mFilename.endsWith(".gz")) {
                        file = new File(dir, this.mFilename + ".gz");
                    } else {
                        if (this.mFilename.length() <= 3) continue;
                        file = new File(dir, this.mFilename.substring(0, this.mFilename.length() - 3));
                    }
                    if (!file.exists()) continue;
                }
                if (!(filename = file.getName()).endsWith(".gz")) {
                    this.mReader = new FileReader(file);
                    break;
                }
                FileInputStream fis = null;
                GZIPInputStream gis = null;
                try {
                    fis = new FileInputStream(file);
                    gis = new GZIPInputStream(fis);
                    this.mReader = new InputStreamReader(gis);
                    break;
                }
                finally {
                    if (this.mReader == null) {
                        try {
                            if (gis != null) {
                                gis.close();
                            } else if (fis != null) {
                                fis.close();
                            }
                        }
                        catch (IOException ee) {}
                    }
                }
            }
        }

        public void close() throws IOException {
            if (this.mReader != null) {
                this.mReader.close();
                this.mReader = null;
            }
        }

        public int read(char[] cbuf, int offset, int len) throws IOException {
            if (this.mReader == null) {
                return -1;
            }
            int charsRead = this.mReader.read(cbuf, offset, len);
            if (charsRead != -1) {
                return charsRead;
            }
            this.openNextReader();
            if (this.mReader == null) {
                return -1;
            }
            cbuf[offset] = 10;
            if (len == 1 || offset + 1 == cbuf.length) {
                return 1;
            }
            charsRead = this.mReader.read(cbuf, offset + 1, len - 1);
            if (charsRead >= 0) {
                return charsRead + 1;
            }
            return charsRead;
        }
    }

    private static class DataColumn {
        private String mInfile;
        private String mColumn;
        private int mHashCode;

        public DataColumn(String infile, String column) {
            this.mInfile = infile;
            this.mColumn = column;
            String hashStr = this.mInfile + "#" + column;
            this.mHashCode = hashStr.hashCode();
        }

        public String getInfile() {
            return this.mInfile;
        }

        public String getColumn() {
            return this.mColumn;
        }

        public int hashCode() {
            return this.mHashCode;
        }

        public boolean equals(Object obj) {
            DataColumn other = (DataColumn)obj;
            return this.mInfile.equals(other.mInfile) && this.mColumn.equals(other.mColumn);
        }

        public String toString() {
            return this.mInfile + ":" + this.mColumn;
        }
    }
}

