/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.cs.store.file;

import com.zimbra.common.localconfig.LC;
import com.zimbra.common.service.ServiceException;
import com.zimbra.common.util.FileUtil;
import com.zimbra.common.util.SystemUtil;
import com.zimbra.common.util.ZimbraLog;
import com.zimbra.cs.localconfig.DebugConfig;
import com.zimbra.cs.mailbox.Mailbox;
import com.zimbra.cs.store.Blob;
import com.zimbra.cs.store.BlobBuilder;
import com.zimbra.cs.store.BlobInputStream;
import com.zimbra.cs.store.FileDescriptorCache;
import com.zimbra.cs.store.IncomingDirectory;
import com.zimbra.cs.store.MailboxBlob;
import com.zimbra.cs.store.StagedBlob;
import com.zimbra.cs.store.StorageCallback;
import com.zimbra.cs.store.StoreManager;
import com.zimbra.cs.store.UncompressedFileCache;
import com.zimbra.cs.store.file.Volume;
import com.zimbra.cs.store.file.VolumeBlob;
import com.zimbra.cs.store.file.VolumeBlobBuilder;
import com.zimbra.cs.store.file.VolumeMailboxBlob;
import com.zimbra.cs.store.file.VolumeStagedBlob;
import com.zimbra.znative.IO;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

public class FileBlobStore
extends StoreManager {
    public void startup() throws IOException, ServiceException {
        Volume.reloadVolumes();
        IncomingDirectory.startSweeper();
        String uncompressedPath = LC.zimbra_tmp_directory.value() + File.separator + "uncompressed";
        FileUtil.ensureDirExists(uncompressedPath);
        UncompressedFileCache<String> ufcache = new UncompressedFileCache(uncompressedPath).startup();
        BlobInputStream.setFileDescriptorCache(new FileDescriptorCache(ufcache).loadSettings());
    }

    public void shutdown() {
        IncomingDirectory.stopSweeper();
    }

    private Blob getUniqueIncomingBlob() throws IOException, ServiceException {
        Volume volume = Volume.getCurrentMessageVolume();
        IncomingDirectory incdir = volume.getIncomingDirectory();
        if (incdir == null) {
            throw ServiceException.FAILURE("storing blob to volume without incoming directory: " + volume.getName(), null);
        }
        File f = incdir.getNewIncomingFile();
        FileBlobStore.ensureParentDirExists(f);
        return new VolumeBlob(f, volume.getId());
    }

    public BlobBuilder getBlobBuilder() throws IOException, ServiceException {
        Blob blob = this.getUniqueIncomingBlob();
        return new VolumeBlobBuilder(blob);
    }

    public Blob storeIncoming(InputStream in, long sizeHint, StorageCallback callback, boolean storeAsIs) throws IOException, ServiceException {
        BlobBuilder builder = this.getBlobBuilder().setSizeHint(sizeHint).setStorageCallback(callback);
        builder.disableCompression(storeAsIs).disableDigest(storeAsIs);
        return builder.init().append(in).finish();
    }

    public VolumeStagedBlob stage(InputStream in, long actualSize, StorageCallback callback, Mailbox mbox) throws IOException, ServiceException {
        Blob blob = this.storeIncoming(in, actualSize, callback);
        return new VolumeStagedBlob(mbox, (VolumeBlob)blob).markStagedDirectly();
    }

    public VolumeStagedBlob stage(Blob blob, Mailbox mbox) throws IOException {
        return new VolumeStagedBlob(mbox, (VolumeBlob)blob);
    }

    public VolumeMailboxBlob copy(MailboxBlob src, Mailbox destMbox, int destMsgId, int destRevision) throws IOException, ServiceException {
        Volume volume = Volume.getCurrentMessageVolume();
        return this.copy(src.getLocalBlob(), destMbox, destMsgId, destRevision, volume);
    }

    public VolumeMailboxBlob copy(Blob src, Mailbox destMbox, int destMsgId, int destRevision, short destVolumeId) throws IOException, ServiceException {
        Volume volume = Volume.getById(destVolumeId);
        return this.copy(src, destMbox, destMsgId, destRevision, volume);
    }

    private VolumeMailboxBlob copy(Blob src, Mailbox destMbox, int destMsgId, int destRevision, Volume destVolume) throws IOException, ServiceException {
        boolean destCompressed;
        File srcFile = src.getFile();
        if (!srcFile.exists()) {
            throw new IOException(srcFile.getPath() + " does not exist");
        }
        String srcPath = src.getPath();
        File dest = this.getMailboxBlobFile(destMbox, destMsgId, destRevision, destVolume.getId(), false);
        String destPath = dest.getAbsolutePath();
        if (ZimbraLog.store.isDebugEnabled()) {
            long srcSize = srcFile.length();
            long srcRawSize = src.getRawSize();
            ZimbraLog.store.debug("Copying %s (size=%d, raw size=%d) to %s for mailbox %d, id %d.", srcPath, srcSize, srcRawSize, destPath, destMbox.getId(), destMsgId);
        }
        FileBlobStore.ensureParentDirExists(dest);
        if (destVolume.getCompressBlobs()) {
            if (src.isCompressed() || srcFile.length() <= destVolume.getCompressionThreshold()) {
                FileUtil.copy(srcFile, dest, !DebugConfig.disableMessageStoreFsync);
                destCompressed = src.isCompressed();
            } else {
                FileUtil.compress(srcFile, dest, !DebugConfig.disableMessageStoreFsync);
                destCompressed = true;
            }
        } else {
            if (src.isCompressed()) {
                FileUtil.uncompress(srcFile, dest, !DebugConfig.disableMessageStoreFsync);
            } else {
                FileUtil.copy(srcFile, dest, !DebugConfig.disableMessageStoreFsync);
            }
            destCompressed = false;
        }
        VolumeBlob newBlob = (VolumeBlob)new VolumeBlob(dest, destVolume.getId()).copyCachedDataFrom(src).setCompressed(destCompressed);
        return new VolumeMailboxBlob(destMbox, destMsgId, destRevision, destVolume.getLocator(), newBlob);
    }

    public VolumeMailboxBlob link(MailboxBlob src, Mailbox destMbox, int destMsgId, int destRevision) throws IOException, ServiceException {
        Volume volume = Volume.getCurrentMessageVolume();
        return this.link(src.getLocalBlob(), destMbox, destMsgId, destRevision, volume.getId());
    }

    public VolumeMailboxBlob link(StagedBlob src, Mailbox destMbox, int destMsgId, int destRevision) throws IOException, ServiceException {
        Volume volume = Volume.getCurrentMessageVolume();
        VolumeBlob blob = ((VolumeStagedBlob)src).getLocalBlob();
        return this.link(blob, destMbox, destMsgId, destRevision, volume.getId());
    }

    public VolumeMailboxBlob link(Blob src, Mailbox destMbox, int destMsgId, int destRevision, short destVolumeId) throws IOException, ServiceException {
        File srcFile = src.getFile();
        if (!srcFile.exists()) {
            throw new IOException(srcFile.getPath() + " does not exist.");
        }
        File dest = this.getMailboxBlobFile(destMbox, destMsgId, destRevision, destVolumeId, false);
        String srcPath = src.getPath();
        String destPath = dest.getAbsolutePath();
        if (ZimbraLog.store.isDebugEnabled()) {
            long srcSize = srcFile.length();
            long srcRawSize = src.getRawSize();
            ZimbraLog.store.debug("Linking %s (size=%d, raw size=%d) to %s for mailbox %d, id %d.", srcPath, srcSize, srcRawSize, destPath, destMbox.getId(), destMsgId);
        }
        FileBlobStore.ensureParentDirExists(dest);
        short srcVolumeId = ((VolumeBlob)src).getVolumeId();
        if (srcVolumeId == destVolumeId) {
            try {
                IO.link((String)srcPath, (String)destPath);
            }
            catch (IOException e) {
                if (dest.exists()) {
                    File destTmp;
                    File destBak = new File(destPath + ".bak");
                    ZimbraLog.store.warn("Destination file exists.  Backing up to " + destBak.getAbsolutePath());
                    if (destBak.exists()) {
                        String bak = destBak.getAbsolutePath();
                        ZimbraLog.store.warn(bak + " already exists.  Deleting to make room for new backup file");
                        if (!destBak.delete()) {
                            ZimbraLog.store.warn("Unable to delete " + bak);
                            throw e;
                        }
                    }
                    if (!(destTmp = new File(destPath)).renameTo(destBak)) {
                        ZimbraLog.store.warn("Can't rename " + destTmp.getAbsolutePath() + " to .bak");
                        throw e;
                    }
                    IO.link((String)srcPath, (String)destPath);
                }
                throw e;
            }
        } else {
            FileUtil.copy(srcFile, dest, !DebugConfig.disableMessageStoreFsync);
        }
        String destLocator = Short.toString(destVolumeId);
        VolumeBlob vblob = (VolumeBlob)new VolumeBlob(dest, destVolumeId).copyCachedDataFrom(src);
        return new VolumeMailboxBlob(destMbox, destMsgId, destRevision, destLocator, vblob);
    }

    public VolumeMailboxBlob renameTo(StagedBlob src, Mailbox destMbox, int destMsgId, int destRevision) throws IOException, ServiceException {
        short srcVolumeId;
        Volume volume = Volume.getCurrentMessageVolume();
        VolumeBlob blob = ((VolumeStagedBlob)src).getLocalBlob();
        File srcFile = blob.getFile();
        String srcPath = srcFile.getAbsolutePath();
        if (!srcFile.exists()) {
            throw new IOException(srcFile.getPath() + " does not exist.");
        }
        File destFile = this.getMailboxBlobFile(destMbox, destMsgId, destRevision, volume.getId(), false);
        String destPath = destFile.getAbsolutePath();
        FileBlobStore.ensureParentDirExists(destFile);
        if (ZimbraLog.store.isDebugEnabled()) {
            long srcSize = srcFile.length();
            long srcRawSize = blob.getRawSize();
            ZimbraLog.store.debug("Renaming %s (size=%d, raw size=%d) to %s for mailbox %d, id %d.", srcPath, srcSize, srcRawSize, destPath, destMbox.getId(), destMsgId);
        }
        if ((srcVolumeId = blob.getVolumeId()) == volume.getId()) {
            boolean renamed = srcFile.renameTo(destFile);
            if (SystemUtil.ON_WINDOWS && !renamed && destFile.exists()) {
                destFile.delete();
                renamed = srcFile.renameTo(destFile);
            }
            if (!renamed) {
                throw new IOException("Unable to rename " + srcPath + " to " + destPath);
            }
        } else {
            FileUtil.copy(srcFile, destFile, !DebugConfig.disableMessageStoreFsync);
            srcFile.delete();
        }
        VolumeBlob vblob = (VolumeBlob)new VolumeBlob(destFile, volume.getId()).copyCachedDataFrom(blob);
        return new VolumeMailboxBlob(destMbox, destMsgId, destRevision, volume.getLocator(), vblob);
    }

    public boolean delete(MailboxBlob mblob) throws IOException {
        if (mblob == null) {
            return false;
        }
        return this.deleteFile(mblob.getLocalBlob().getFile());
    }

    public boolean delete(StagedBlob staged) throws IOException {
        VolumeStagedBlob vsb = (VolumeStagedBlob)staged;
        if (staged == null || !vsb.wasStagedDirectly()) {
            return false;
        }
        return this.deleteFile(vsb.getLocalBlob().getFile());
    }

    public boolean delete(Blob blob) throws IOException {
        if (blob == null) {
            return false;
        }
        return this.deleteFile(blob.getFile());
    }

    private boolean deleteFile(File file) throws IOException {
        if (file == null) {
            return false;
        }
        ZimbraLog.store.debug("Deleting %s.", file.getPath());
        boolean deleted = file.delete();
        if (deleted) {
            return true;
        }
        if (!file.exists()) {
            return false;
        }
        throw new IOException("Unable to delete blob file " + file.getAbsolutePath());
    }

    public MailboxBlob getMailboxBlob(Mailbox mbox, int msgId, int revision, String locator) throws ServiceException {
        short volumeId = Short.parseShort(locator);
        File file = this.getMailboxBlobFile(mbox, msgId, revision, volumeId, true);
        if (file == null) {
            return null;
        }
        return new VolumeMailboxBlob(mbox, msgId, revision, locator, new VolumeBlob(file, volumeId));
    }

    public InputStream getContent(MailboxBlob mboxBlob) throws IOException {
        if (mboxBlob == null) {
            return null;
        }
        return this.getContent(mboxBlob.getLocalBlob());
    }

    public InputStream getContent(Blob blob) throws IOException {
        if (blob == null) {
            return null;
        }
        return new BlobInputStream(blob);
    }

    public boolean deleteStore(Mailbox mbox) throws IOException {
        for (Volume vol : Volume.getAll()) {
            FileUtil.deleteDir(new File(vol.getMessageRootDir(mbox.getId())));
        }
        return true;
    }

    private File getMailboxBlobFile(Mailbox mbox, int msgId, int revision, short volumeId, boolean check) throws ServiceException {
        File file = new File(FileBlobStore.getBlobPath(mbox, msgId, revision, volumeId));
        if (!check || file.exists()) {
            return file;
        }
        file = new File(FileBlobStore.getBlobPath(mbox, msgId, -1, volumeId));
        return file.exists() ? file : null;
    }

    public static String getBlobPath(Mailbox mbox, int msgId, int revision, short volumeId) throws ServiceException {
        Volume vol = Volume.getById(volumeId);
        String path = vol.getBlobDir(mbox.getId(), msgId);
        int buflen = path.length() + 15 + (revision < 0 ? 0 : 11);
        StringBuffer sb = new StringBuffer(buflen);
        sb.append(path).append(File.separator);
        FileBlobStore.appendFilename(sb, msgId, revision);
        return sb.toString();
    }

    public static void appendFilename(StringBuffer sb, int itemId, int revision) {
        sb.append(itemId);
        if (revision >= 0) {
            sb.append('-').append(revision);
        }
        sb.append(".msg");
    }

    public static String getFilename(int itemId, int revision) {
        StringBuffer buf = new StringBuffer();
        FileBlobStore.appendFilename(buf, itemId, revision);
        return buf.toString();
    }

    private static void ensureDirExists(File dir) throws IOException {
        if (!FileUtil.mkdirs(dir)) {
            throw new IOException("Unable to create blob store directory " + dir.getAbsolutePath());
        }
    }

    private static void ensureParentDirExists(File file) throws IOException {
        FileBlobStore.ensureDirExists(file.getParentFile());
    }
}

