/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.cs.redolog;

import com.zimbra.common.util.FileUtil;
import com.zimbra.common.util.Log;
import com.zimbra.common.util.LogFactory;
import com.zimbra.cs.redolog.RedoLogManager;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.TimeZone;

public class RolloverManager {
    private static Log mLog = LogFactory.getLog(RolloverManager.class);
    private RedoLogManager mRedoLogMgr;
    private File mRedoLogFile;
    private long mSequence;
    private static final String ARCH_FILENAME_PREFIX = "redo-";
    private static final String TEMP_FILENAME_PREFIX = "~tmp-redo-";
    private static final String SEQUENCE_PREFIX = "-seq";
    private static final String FILENAME_SUFFIX = ".log";
    private static final String TIMESTAMP_FORMAT = "yyyyMMdd.HHmmss.SSS";

    public RolloverManager(RedoLogManager redoLogMgr, File redolog) {
        this.mRedoLogMgr = redoLogMgr;
        this.mRedoLogFile = redolog;
        this.mSequence = 0L;
    }

    public void crashRecovery() throws IOException {
        File[] logs = this.mRedoLogFile.getParentFile().listFiles(new TempLogFilenameFilter());
        if (logs.length > 0) {
            FileUtil.sortFilesByModifiedTime(logs);
            if (!this.mRedoLogFile.exists()) {
                File mostRecent = logs[logs.length - 1];
                String name = mostRecent.getName();
                String currName = this.mRedoLogFile.getName();
                if (!mostRecent.renameTo(this.mRedoLogFile)) {
                    throw new IOException("Unable to rename " + name + " to " + currName);
                }
                mLog.info("Renamed " + name + " to " + currName);
                logs[logs.length - 1] = null;
            }
            int numErrors = 0;
            for (int i = 0; i < logs.length && logs[i] != null; ++i) {
                File log = logs[i];
                String oldName = log.getName();
                String newName = oldName + ".bak";
                File newLog = new File(log.getParentFile(), newName);
                if (log.renameTo(newLog)) {
                    mLog.info("Renamed " + oldName + " to " + newName);
                    continue;
                }
                ++numErrors;
                mLog.error("Unable to rename " + oldName + " to " + newName);
            }
            if (numErrors > 0) {
                throw new IOException("Error(s) occurred while renaming temporary redo log files");
            }
        }
    }

    public static File[] getArchiveLogs(File archiveDir) {
        return RolloverManager.getArchiveLogs(archiveDir, Long.MIN_VALUE, Long.MAX_VALUE);
    }

    public static File[] getArchiveLogs(File archiveDir, long from) {
        return RolloverManager.getArchiveLogs(archiveDir, from, Long.MAX_VALUE);
    }

    public static File[] getArchiveLogs(File archiveDir, final long from, final long to) {
        File[] logs = archiveDir.listFiles(new FilenameFilter(){

            public boolean accept(File dir, String name) {
                long seq;
                return name.indexOf(RolloverManager.ARCH_FILENAME_PREFIX) == 0 && name.lastIndexOf(RolloverManager.FILENAME_SUFFIX) == name.length() - RolloverManager.FILENAME_SUFFIX.length() && from <= (seq = RolloverManager.getSeqForFile(new File(dir, name))) && seq <= to;
            }
        });
        if (logs != null && logs.length > 0) {
            RolloverManager.sortArchiveLogFiles(logs);
        }
        return logs;
    }

    public static void sortArchiveLogFiles(File[] files) {
        ArchiveLogFilenameComparator comp = new ArchiveLogFilenameComparator();
        Arrays.sort(files, comp);
    }

    public static long getSeqForFile(File f) {
        String fname = f.getName();
        int start = fname.lastIndexOf(SEQUENCE_PREFIX);
        if (start == -1) {
            return -1L;
        }
        int end = fname.indexOf(FILENAME_SUFFIX, start += SEQUENCE_PREFIX.length());
        if (end == -1) {
            return -1L;
        }
        try {
            String val = fname.substring(start, end);
            return Long.parseLong(val);
        }
        catch (StringIndexOutOfBoundsException se) {
            return -1L;
        }
        catch (NumberFormatException ne) {
            return -1L;
        }
    }

    public static long getEndTimeForFile(File f) {
        SimpleDateFormat fmt = new SimpleDateFormat(TIMESTAMP_FORMAT);
        fmt.setLenient(false);
        int prefixLen = ARCH_FILENAME_PREFIX.length();
        try {
            Date d = fmt.parse(f.getName().substring(prefixLen));
            return d.getTime();
        }
        catch (ParseException e) {
            return f.lastModified();
        }
    }

    public static String toArchiveLogFilename(Date date, long seq) {
        StringBuilder fname = new StringBuilder(ARCH_FILENAME_PREFIX);
        SimpleDateFormat fmt = new SimpleDateFormat(TIMESTAMP_FORMAT);
        fmt.setTimeZone(TimeZone.getTimeZone("GMT"));
        fname.append(fmt.format(date));
        fname.append(SEQUENCE_PREFIX).append(seq);
        fname.append(FILENAME_SUFFIX);
        return fname.toString();
    }

    public File getRolloverFile(long seq) {
        String fname = RolloverManager.toArchiveLogFilename(new Date(), seq);
        File destDir = this.mRedoLogMgr.getRolloverDestDir();
        if (!(destDir.exists() || destDir.mkdir() || destDir.exists())) {
            mLog.error("Unable to create rollover destination directory " + destDir.getAbsolutePath());
        }
        return new File(destDir, fname);
    }

    public String getTempFilename(long seq) {
        StringBuilder fname = new StringBuilder(TEMP_FILENAME_PREFIX);
        SimpleDateFormat fmt = new SimpleDateFormat(TIMESTAMP_FORMAT);
        fmt.setTimeZone(TimeZone.getTimeZone("GMT"));
        fname.append(fmt.format(new Date()));
        fname.append(SEQUENCE_PREFIX).append(seq);
        fname.append(FILENAME_SUFFIX);
        return fname.toString();
    }

    public synchronized long getCurrentSequence() {
        return this.mSequence;
    }

    public synchronized void initSequence(long seq) {
        this.mSequence = seq;
    }

    public synchronized long incrementSequence() {
        this.mSequence = this.mSequence < Long.MAX_VALUE ? ++this.mSequence : 0L;
        return this.mSequence;
    }

    private static class TempLogFilenameFilter
    implements FilenameFilter {
        private TempLogFilenameFilter() {
        }

        public boolean accept(File dir, String name) {
            return name.indexOf(RolloverManager.TEMP_FILENAME_PREFIX) == 0 && name.lastIndexOf(RolloverManager.FILENAME_SUFFIX) == name.length() - RolloverManager.FILENAME_SUFFIX.length();
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static class ArchiveLogFilenameComparator
    implements Comparator<File> {
        private ArchiveLogFilenameComparator() {
        }

        @Override
        public int compare(File f1, File f2) {
            long t2;
            long t1 = RolloverManager.getSeqForFile(f1);
            if (t1 < (t2 = RolloverManager.getSeqForFile(f2))) {
                return -1;
            }
            if (t1 > t2) {
                return 1;
            }
            t1 = RolloverManager.getEndTimeForFile(f1);
            if (t1 < (t2 = RolloverManager.getEndTimeForFile(f2))) {
                return -1;
            }
            if (t1 > t2) {
                return 1;
            }
            return 0;
        }
    }
}

