/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.cs.db;

import com.zimbra.cs.db.DbUtil;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class JdbcClient {
    private static final String OPT_HELP = "h";
    private static final String OPT_USER = "u";
    private static final String OPT_PASSWORD = "p";
    private static final String OPT_JDBC_URL = "j";
    private static final String OPT_DRIVER_CLASS = "c";
    private static final String OPT_BATCH = "B";
    private static final String OPT_SKIP_COLUMN_NAMES = "N";
    private static final String NULL = "NULL";
    private static final Pattern PAT_QUIT = Pattern.compile("\\s*quit;?", 2);
    private static final Pattern PAT_EXIT = Pattern.compile("\\s*exit;?", 2);
    private static final Pattern PAT_SELECT = Pattern.compile("^\\s*SELECT", 2);
    private static final Pattern PAT_SEMICOLON = Pattern.compile("([^;]*);(.*)");
    private Options mOptions = new Options();
    private String mDriverClass;
    private String mUser;
    private String mPassword;
    private String mJdbcUrl;
    private boolean mBatch = false;
    private boolean mShowColumnNames = true;

    private JdbcClient(String[] args) {
        Option opt = new Option(OPT_HELP, "help", false, "Display this help message.");
        this.mOptions.addOption(opt);
        opt = new Option(OPT_DRIVER_CLASS, "driver-class", true, "JDBC driver class name.");
        opt.setRequired(true);
        this.mOptions.addOption(opt);
        opt = new Option(OPT_USER, "user", true, "User name.");
        opt.setRequired(true);
        this.mOptions.addOption(opt);
        opt = new Option(OPT_PASSWORD, "password", true, "Password.");
        opt.setRequired(true);
        this.mOptions.addOption(opt);
        opt = new Option(OPT_JDBC_URL, "jdbc-url", true, "JDBC URL used for connecting to the database server.");
        opt.setRequired(true);
        this.mOptions.addOption(opt);
        this.mOptions.addOption(OPT_BATCH, "batch", false, "Prints results without formatting, separated by tabs.");
        this.mOptions.addOption(OPT_SKIP_COLUMN_NAMES, "skip-column-names", false, "Don't write column names in results.");
        CommandLine cl = null;
        try {
            GnuParser parser = new GnuParser();
            cl = parser.parse(this.mOptions, args);
        }
        catch (ParseException e) {
            this.usage(e);
            System.exit(1);
        }
        if (cl.hasOption(OPT_HELP)) {
            this.usage(null);
            System.exit(0);
        }
        this.mDriverClass = cl.getOptionValue(OPT_DRIVER_CLASS);
        this.mUser = cl.getOptionValue(OPT_USER);
        this.mPassword = cl.getOptionValue(OPT_PASSWORD);
        this.mJdbcUrl = cl.getOptionValue(OPT_JDBC_URL);
        if (cl.hasOption(OPT_BATCH)) {
            this.mBatch = true;
        }
        if (cl.hasOption(OPT_SKIP_COLUMN_NAMES)) {
            this.mShowColumnNames = false;
        }
    }

    private void run() {
        try {
            Class.forName(this.mDriverClass);
        }
        catch (Throwable t) {
            System.err.println("Unable to load driver '" + this.mDriverClass + "': " + t);
            System.exit(1);
        }
        Connection conn = null;
        try {
            conn = DriverManager.getConnection(this.mJdbcUrl, this.mUser, this.mPassword);
        }
        catch (SQLException e) {
            System.err.println("Unable to connect to " + this.mJdbcUrl + " using " + this.mDriverClass + ": " + e);
            System.exit(1);
        }
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(System.in));
            String line = "";
            StringBuilder b = new StringBuilder();
            while (line != null) {
                Matcher quitMatcher = PAT_QUIT.matcher(line);
                Matcher exitMatcher = PAT_EXIT.matcher(line);
                if (quitMatcher.matches() || exitMatcher.matches()) break;
                Matcher semicolonMatcher = PAT_SEMICOLON.matcher(line = DbUtil.removeComments(line));
                if (semicolonMatcher.matches()) {
                    b.append(semicolonMatcher.group(1));
                    this.runSql(conn, b.toString());
                    b = new StringBuilder();
                    line = semicolonMatcher.group(2);
                    continue;
                }
                b.append(line);
                line = reader.readLine();
            }
            if (b.length() > 0) {
                this.runSql(conn, b.toString());
            }
        }
        catch (IOException e) {
            System.err.println(e);
            System.exit(1);
        }
        try {
            conn.close();
        }
        catch (SQLException e) {
            System.err.println(e);
            System.exit(1);
        }
    }

    private void runSql(Connection conn, String sql) {
        Matcher m = PAT_SELECT.matcher(sql);
        if (m.find()) {
            try {
                Statement stmt = conn.createStatement();
                ResultSet rs = stmt.executeQuery(sql);
                ResultSetMetaData md = rs.getMetaData();
                int colCount = md.getColumnCount();
                ArrayList<Object[]> firstRows = new ArrayList<Object[]>();
                int rowCount = 0;
                int[] colWidths = new int[colCount];
                if (this.mShowColumnNames) {
                    for (int i = 0; i < colCount; ++i) {
                        String name = md.getColumnLabel(i + 1);
                        if (name.length() <= colWidths[i]) continue;
                        colWidths[i] = name.length();
                    }
                }
                while (rowCount < 1000 && rs.next()) {
                    Object[] row = this.getCurrentRow(rs);
                    for (int i = 0; i < colCount; ++i) {
                        int width;
                        Object o = row[i];
                        int n = width = o == null ? NULL.length() : o.toString().length();
                        if (width <= colWidths[i]) continue;
                        colWidths[i] = width;
                    }
                    firstRows.add(row);
                    ++rowCount;
                }
                if (!this.mBatch && this.mShowColumnNames) {
                    this.printDivider(colWidths);
                }
                if (this.mShowColumnNames) {
                    Object[] colNames = new String[colCount];
                    for (int i = 0; i < colCount; ++i) {
                        colNames[i] = md.getColumnLabel(i + 1);
                    }
                    this.printRow(colNames, colWidths);
                }
                if (!this.mBatch) {
                    this.printDivider(colWidths);
                }
                for (Object[] row : firstRows) {
                    this.printRow(row, colWidths);
                }
                while (rs.next()) {
                    Object[] row = this.getCurrentRow(rs);
                    this.printRow(row, colWidths);
                }
                if (!this.mBatch) {
                    this.printDivider(colWidths);
                }
                rs.close();
                stmt.close();
            }
            catch (SQLException e) {
                e.printStackTrace();
                System.err.println(e.getMessage());
            }
        } else {
            try {
                Statement stmt = conn.createStatement();
                int numRows = stmt.executeUpdate(sql);
                stmt.close();
                System.out.println("Updated " + numRows + " rows");
            }
            catch (SQLException e) {
                System.err.println(e.getMessage());
            }
        }
    }

    private void printDivider(int[] colWidths) {
        StringBuilder b = new StringBuilder();
        for (int i = 0; i < colWidths.length; ++i) {
            b.append('+');
            for (int j = 0; j < colWidths[i] + 2; ++j) {
                b.append('-');
            }
        }
        b.append('+');
        System.out.println(b);
    }

    private Object[] getCurrentRow(ResultSet rs) throws SQLException {
        ResultSetMetaData md = rs.getMetaData();
        int colCount = md.getColumnCount();
        Object[] row = new Object[colCount];
        for (int i = 0; i < colCount; ++i) {
            row[i] = rs.getObject(i + 1);
        }
        return row;
    }

    private void printRow(Object[] row, int[] colWidths) {
        StringBuilder b = new StringBuilder();
        if (!this.mBatch) {
            b.append("| ");
        }
        String delimiter = this.mBatch ? "\t" : " | ";
        for (int i = 0; i < row.length; ++i) {
            Object o;
            if (i > 0) {
                b.append(delimiter);
            }
            String s = (o = row[i]) == null ? NULL : o.toString();
            b.append(s);
            if (this.mBatch || s.length() >= colWidths[i]) continue;
            for (int j = 0; j < colWidths[i] - s.length(); ++j) {
                b.append(" ");
            }
        }
        if (!this.mBatch) {
            b.append(" |");
        }
        System.out.println(b);
    }

    private void usage(ParseException e) {
        if (e != null) {
            System.err.println((Object)((Object)e) + "\n");
        }
        PrintWriter pw = new PrintWriter(System.err, true);
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(pw, formatter.getWidth(), "zmsql <options>", null, this.mOptions, formatter.getLeftPadding(), formatter.getDescPadding(), "");
    }

    public static void main(String[] args) {
        JdbcClient app = new JdbcClient(args);
        app.run();
    }
}

