/*
 * Decompiled with CFR 0.152.
 */
package com.zimbra.common.util;

import com.zimbra.common.service.ServiceException;
import java.text.DateFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DateUtil {
    public static final String ZIMBRA_LDAP_GENERALIZED_TIME_FORMAT = "yyyyMMddHHmmss'Z'";
    private static final String[] DAY_NAME = new String[]{"", "Sun, ", "Mon, ", "Tue, ", "Wed, ", "Thu, ", "Fri, ", "Sat, "};
    private static final String[] MONTH_NAME = new String[]{"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
    private static final String[][] ZONE_INFO = new DateFormatSymbols().getZoneStrings();
    private static final Map<String, Integer> MONTH_NUMBER = new HashMap<String, Integer>(16);
    private static final Map<String, String> KNOWN_ZONES;
    private static final String ABSDATE_YFIRST_PATTERN = "(\\d{4})[/-](\\d{1,2})[/-](\\d{1,2})";
    private static final String ABSDATE_YLAST_PATTERN = "(\\d{1,2})[/-](\\d{1,2})[/-](\\d{4})";
    private static final String RELDATE_PATTERN = "([mp+-]?)([0-9]+)([mhdwy][a-z]*)?";
    private static final String ABS_MILLISECS_PATTERN = "\\d+";
    private static final Pattern sAbsYFirstPattern;
    private static final Pattern sAbsYLastPattern;
    private static final Pattern sRelDatePattern;
    private static final Pattern sAbsMillisecsDatePattern;

    public static String toGeneralizedTime(Date date) {
        SimpleDateFormat fmt = new SimpleDateFormat(ZIMBRA_LDAP_GENERALIZED_TIME_FORMAT);
        TimeZone tz = fmt.getCalendar().getTimeZone();
        Date gmtDate = tz.inDaylightTime(date) ? new Date(date.getTime() - (long)(tz.getRawOffset() + tz.getDSTSavings())) : new Date(date.getTime() - (long)tz.getRawOffset());
        return fmt.format(gmtDate);
    }

    public static Date parseGeneralizedTime(String time) {
        return DateUtil.parseGeneralizedTime(time, true);
    }

    public static Date parseGeneralizedTime(String time, boolean strict) {
        int maxLen = strict ? 15 : 17;
        if (time.length() < 14 || time.length() > maxLen) {
            return null;
        }
        TimeZone tz = time.endsWith("Z") ? TimeZone.getTimeZone("GMT") : TimeZone.getDefault();
        int year = Integer.parseInt(time.substring(0, 4));
        int month = Integer.parseInt(time.substring(4, 6)) - 1;
        int date = Integer.parseInt(time.substring(6, 8));
        int hour = Integer.parseInt(time.substring(8, 10));
        int min = Integer.parseInt(time.substring(10, 12));
        int sec = Integer.parseInt(time.substring(12, 14));
        GregorianCalendar calendar = new GregorianCalendar(tz);
        calendar.clear();
        calendar.set(year, month, date, hour, min, sec);
        return calendar.getTime();
    }

    public static String toISO8601(Date date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
        String result = format.format(date);
        result = result.substring(0, result.length() - 2) + ":" + result.substring(result.length() - 2);
        return result;
    }

    public static String toImapDateTime(Date date) {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTime(date);
        int tzoffset = (cal.get(15) + cal.get(16)) / 60000;
        char tzsign = tzoffset > 0 ? (char)'+' : '-';
        tzoffset = Math.abs(tzoffset);
        StringBuilder sb = new StringBuilder(40);
        DateUtil.append2DigitNumber(sb, cal.get(5)).append('-');
        sb.append(MONTH_NAME[cal.get(2)]).append('-');
        sb.append(cal.get(1)).append(' ');
        DateUtil.append2DigitNumber(sb, cal.get(11)).append(':');
        DateUtil.append2DigitNumber(sb, cal.get(12)).append(':');
        DateUtil.append2DigitNumber(sb, cal.get(13)).append(' ');
        sb.append(tzsign);
        DateUtil.append2DigitNumber(sb, tzoffset / 60);
        DateUtil.append2DigitNumber(sb, tzoffset % 60);
        return sb.toString();
    }

    public static String toRFC822Date(Date date) {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTime(date);
        String tzabbr = DateUtil.getTimezoneAbbreviation(((Calendar)cal).getTimeZone().getID(), cal.get(16) != 0);
        int tzoffset = (cal.get(15) + cal.get(16)) / 60000;
        char tzsign = tzoffset > 0 ? (char)'+' : '-';
        tzoffset = Math.abs(tzoffset);
        StringBuilder sb = new StringBuilder(40);
        sb.append(DAY_NAME[cal.get(7)]);
        DateUtil.append2DigitNumber(sb, cal.get(5)).append(' ');
        sb.append(MONTH_NAME[cal.get(2)]).append(' ');
        sb.append(cal.get(1)).append(' ');
        DateUtil.append2DigitNumber(sb, cal.get(11)).append(':');
        DateUtil.append2DigitNumber(sb, cal.get(12)).append(':');
        DateUtil.append2DigitNumber(sb, cal.get(13)).append(' ');
        sb.append(tzsign);
        DateUtil.append2DigitNumber(sb, tzoffset / 60);
        DateUtil.append2DigitNumber(sb, tzoffset % 60);
        if (tzabbr != null) {
            sb.append(" (").append(tzabbr).append(')');
        }
        return sb.toString();
    }

    private static StringBuilder append2DigitNumber(StringBuilder sb, int number) {
        return sb.append((char)(48 + number / 10)).append((char)(48 + number % 10));
    }

    private static String getTimezoneAbbreviation(String tzid, boolean dst) {
        if (tzid == null) {
            return null;
        }
        for (int tzindex = 0; tzindex < ZONE_INFO.length; ++tzindex) {
            if (!tzid.equalsIgnoreCase(ZONE_INFO[tzindex][0])) continue;
            return dst ? ZONE_INFO[tzindex][4] : ZONE_INFO[tzindex][2];
        }
        return null;
    }

    public static Date parseRFC2822Date(String encoded, Date fallback) {
        if (encoded == null) {
            return fallback;
        }
        GregorianCalendar cal = new GregorianCalendar();
        cal.set(13, 0);
        try {
            int pos = DateUtil.skipCFWS(encoded, 0);
            while (encoded.charAt(pos) < '0' || encoded.charAt(pos) > '9') {
                pos = DateUtil.skipCFWS(encoded, DateUtil.skipText(encoded, pos, false));
            }
            pos = DateUtil.readCalendarField(encoded, pos, cal, 5);
            pos = DateUtil.readCalendarMonth(encoded, DateUtil.skipCFWS(encoded, pos), cal);
            pos = DateUtil.readCalendarField(encoded, DateUtil.skipCFWS(encoded, pos), cal, 1);
            pos = DateUtil.readCalendarField(encoded, DateUtil.skipCFWS(encoded, pos), cal, 11);
            pos = DateUtil.skipChar(encoded, DateUtil.skipCFWS(encoded, pos), ':');
            pos = DateUtil.readCalendarField(encoded, DateUtil.skipCFWS(encoded, pos), cal, 12);
            if (encoded.charAt(pos = DateUtil.skipCFWS(encoded, pos)) == ':') {
                pos = DateUtil.readCalendarField(encoded, DateUtil.skipCFWS(encoded, pos + 1), cal, 13);
            }
            DateUtil.readCalendarTimeZone(encoded, DateUtil.skipCFWS(encoded, pos), cal);
        }
        catch (ParseException e) {
            return fallback;
        }
        return cal.getTime();
    }

    private static int skipCFWS(String encoded, int start) throws ParseException {
        boolean escaped = false;
        int len = encoded.length();
        int comment = 0;
        for (int i = start; i < len; ++i) {
            char c = encoded.charAt(i);
            if (escaped) {
                escaped = false;
                continue;
            }
            if (c == '(') {
                ++comment;
                continue;
            }
            if (c == ')' && comment > 0) {
                --comment;
                continue;
            }
            if (c == '\\' && comment > 0) {
                escaped = true;
                continue;
            }
            if (c == ' ' || c == '\t' || c == '\r' || c == '\n' || comment != 0) continue;
            return i;
        }
        throw new ParseException("CFWS extends to end of header", start);
    }

    private static int skipChar(String encoded, int start, char skip) throws ParseException {
        if (start >= encoded.length() || encoded.charAt(start) != skip) {
            throw new ParseException("missing expected character '" + skip + "'", start);
        }
        return start + 1;
    }

    private static int skipText(String encoded, int start, boolean canTerminate) throws ParseException {
        int len = encoded.length();
        for (int i = start; i < len; ++i) {
            char c = encoded.charAt(i);
            if (c != '(' && c != ' ' && c != '\t' && c != '\r' && c != '\n') continue;
            if (start == i) {
                throw new ParseException("unexpected zero-length text string", start);
            }
            return i;
        }
        if (!canTerminate) {
            throw new ParseException("skipped text extends to end of header", start);
        }
        return encoded.length();
    }

    private static int readCalendarField(String encoded, int start, Calendar cal, int field) {
        char c;
        int i;
        int value = 0;
        int len = encoded.length();
        for (i = start; i < len && (c = encoded.charAt(i)) >= '0' && c <= '9'; ++i) {
            value = value * 10 + c - 48;
        }
        if (field == 1 && value < 32) {
            value += 2000;
        } else if (field == 1 && value < 1000) {
            value += 1900;
        }
        cal.set(field, value);
        return i;
    }

    private static int readCalendarMonth(String encoded, int start, Calendar cal) throws ParseException {
        int i = DateUtil.skipText(encoded, start, true);
        String monthabbr = encoded.substring(start, i);
        Integer month = MONTH_NUMBER.get(monthabbr.toUpperCase());
        if (month == null) {
            try {
                month = Integer.valueOf(monthabbr) - 1;
            }
            catch (NumberFormatException nfe) {
                // empty catch block
            }
        }
        if (month == null) {
            throw new ParseException("invalid month abbreviation: " + monthabbr, start);
        }
        cal.set(2, month);
        return i;
    }

    private static int readCalendarTimeZone(String encoded, int start, Calendar cal) throws ParseException {
        int i = DateUtil.skipText(encoded, start, true);
        String zone = encoded.substring(start, i);
        int offset = 0;
        if (zone.charAt(0) != '-' && zone.charAt(0) != '+') {
            zone = KNOWN_ZONES.get(zone.toUpperCase());
        }
        if (zone != null) {
            try {
                int parsed = Integer.parseInt(zone.substring(1));
                offset = (zone.charAt(0) == '-' ? -1 : 1) * (parsed / 100 * 60 + parsed % 100) * 60000;
            }
            catch (NumberFormatException nfe) {
                // empty catch block
            }
        }
        cal.set(15, offset);
        cal.set(16, 0);
        return i;
    }

    public static Date parseISO8601Date(String encoded, Date fallback) {
        if (encoded == null) {
            return fallback;
        }
        int length = (encoded = encoded.toUpperCase()).length();
        if (length == 4) {
            encoded = encoded + "-01-01T00:00:00-0000";
        } else if (length == 7) {
            encoded = encoded + "-01T00:00:00-0000";
        } else if (length == 10) {
            encoded = encoded + "T00:00:00-0000";
        } else {
            if (length < 17) {
                return fallback;
            }
            if (encoded.charAt(16) != ':') {
                encoded = encoded.substring(0, 16) + ":00" + encoded.substring(16);
            } else if (length >= 21 && encoded.charAt(19) == '.') {
                int pos;
                for (pos = 20; pos < length && Character.isDigit(encoded.charAt(pos)); ++pos) {
                }
                encoded = encoded.substring(0, 19) + encoded.substring(pos);
            }
        }
        int colon = encoded.lastIndexOf(58);
        if (colon > 19) {
            encoded = encoded.substring(0, colon) + encoded.substring(colon + 1);
        }
        if (encoded.length() == 19) {
            encoded = encoded + "-0000";
        } else if (encoded.endsWith("Z")) {
            encoded = encoded.substring(0, encoded.length() - 1) + "-0000";
        }
        try {
            return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").parse(encoded);
        }
        catch (ParseException e) {
            return fallback;
        }
    }

    public static long parseDateSpecifier(String dateStr, long defaultValue) {
        Date date = DateUtil.parseDateSpecifier(dateStr);
        return date == null ? defaultValue : date.getTime();
    }

    public static Date parseDateSpecifier(String dateStr) {
        try {
            Matcher m = sAbsMillisecsDatePattern.matcher(dateStr);
            if (m.matches()) {
                return new Date(Long.parseLong(dateStr));
            }
            m = sAbsYFirstPattern.matcher(dateStr);
            if (m.matches()) {
                String yearStr = m.group(1);
                String monthStr = m.group(2);
                String dayStr = m.group(3);
                return new SimpleDateFormat("MM/dd/yyyy").parse(monthStr + "/" + dayStr + "/" + yearStr);
            }
            m = sAbsYLastPattern.matcher(dateStr);
            if (m.matches()) {
                String monthStr = m.group(1);
                String dayStr = m.group(2);
                String yearStr = m.group(3);
                return new SimpleDateFormat("MM/dd/yyyy").parse(monthStr + "/" + dayStr + "/" + yearStr);
            }
            m = sRelDatePattern.matcher(dateStr);
            if (m.matches()) {
                String ss = m.group(1);
                int sign = ss == null || ss.equals("") || ss.equals("+") || ss.equals("p") ? 1 : -1;
                int value = Integer.parseInt(m.group(2)) * sign;
                String unitsStr = m.group(3);
                int field = 5;
                if (unitsStr != null && unitsStr.length() > 0) {
                    switch (unitsStr.charAt(0)) {
                        case 'm': {
                            field = unitsStr.length() > 1 && unitsStr.charAt(1) == 'i' ? 12 : 2;
                            break;
                        }
                        case 'h': {
                            field = 10;
                            break;
                        }
                        case 'd': {
                            field = 5;
                            break;
                        }
                        case 'w': {
                            field = 3;
                            break;
                        }
                        case 'y': {
                            field = 1;
                        }
                    }
                }
                GregorianCalendar cal = new GregorianCalendar();
                cal.setTime(new Date());
                ((Calendar)cal).add(field, value);
                return cal.getTime();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    public static long getTimeInterval(String value, long defaultValue) {
        if (value == null || value.length() == 0) {
            return defaultValue;
        }
        try {
            char units = value.charAt(value.length() - 1);
            if (units >= '0' && units <= '9') {
                return Long.parseLong(value) * 1000L;
            }
            if (value.endsWith("ms")) {
                return Long.parseLong(value.substring(0, value.length() - 2));
            }
            long n = Long.parseLong(value.substring(0, value.length() - 1));
            switch (units) {
                case 'd': {
                    n *= 86400000L;
                    break;
                }
                case 'h': {
                    n *= 3600000L;
                    break;
                }
                case 'm': {
                    n *= 60000L;
                    break;
                }
                case 's': {
                    n *= 1000L;
                    break;
                }
                default: {
                    return defaultValue;
                }
            }
            return n;
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public static long getTimeInterval(String value) throws ServiceException {
        if (value == null || value.length() == 0) {
            throw ServiceException.FAILURE("no value", null);
        }
        try {
            char units = value.charAt(value.length() - 1);
            if (units >= '0' && units <= '9') {
                return Long.parseLong(value) * 1000L;
            }
            if (value.endsWith("ms")) {
                return Long.parseLong(value.substring(0, value.length() - 2));
            }
            long n = Long.parseLong(value.substring(0, value.length() - 1));
            switch (units) {
                case 'd': {
                    n *= 86400000L;
                    break;
                }
                case 'h': {
                    n *= 3600000L;
                    break;
                }
                case 'm': {
                    n *= 60000L;
                    break;
                }
                case 's': {
                    n *= 1000L;
                    break;
                }
                default: {
                    throw ServiceException.FAILURE("unknown unit", null);
                }
            }
            return n;
        }
        catch (NumberFormatException e) {
            throw ServiceException.FAILURE("invalid format", e);
        }
    }

    public static long getTimeIntervalSecs(String value, long defaultValue) {
        if (value == null || value.length() == 0) {
            return defaultValue;
        }
        try {
            char units = value.charAt(value.length() - 1);
            if (units >= '0' && units <= '9') {
                return Long.parseLong(value);
            }
            if (value.endsWith("ms")) {
                long ms = Long.parseLong(value.substring(0, value.length() - 2));
                return Math.round((float)ms / 1000.0f);
            }
            long n = Long.parseLong(value.substring(0, value.length() - 1));
            switch (units) {
                case 'd': {
                    n *= 86400L;
                    break;
                }
                case 'h': {
                    n *= 3600L;
                    break;
                }
                case 'm': {
                    n *= 60L;
                    break;
                }
                case 's': {
                    break;
                }
                default: {
                    return defaultValue;
                }
            }
            return n;
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public static void main(String[] args) {
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("12/25/1998", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("1989/12/25", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("1day", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+1day", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+10day", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+60minute", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+1week", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+1month", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("+1year", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("-1day", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("1132276598000", 0L)));
        System.out.println("date = " + new Date(DateUtil.parseDateSpecifier("p10day", 0L)));
        System.out.println("iso = " + DateUtil.parseISO8601Date("2009-11-20", new Date()));
        System.out.println("iso = " + DateUtil.parseISO8601Date("2009-11-20T13:55:49Z", new Date()));
        System.out.println("iso = " + DateUtil.parseISO8601Date("2009-11-20T13:55:49.823", new Date()));
        System.out.println("iso = " + DateUtil.parseISO8601Date("2009-11-20t13:55:49.724z", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("01 May 2007 09:27 -0600", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("01 Apr 2005 18:20:24 -0800", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("Fri, 10 Oct 2003 12:52:52 -0700", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("Wed, 27 Apr 2005 11:14:18 -0700 (PDT)", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("Fri, 8 Feb 2008 00:56:02 -0500 (EST)", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("Wed, 27 Apr 2005 15:37:37 PDT", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("21 Nov 97 09:55:06 GMT", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("Tue,  1 May 2007 09:41(() )():(() )()26 -0700", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("21 Nov 97 09::06 GMT", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("21 nOV 97 09:55:06 R", new Date()));
        System.out.println(DateUtil.parseRFC2822Date("21 11 06 09:55:06 GMT", new Date()));
        System.out.println(DateUtil.toImapDateTime(new Date()));
    }

    static {
        for (int i = 0; i < MONTH_NAME.length; ++i) {
            MONTH_NUMBER.put(MONTH_NAME[i].toUpperCase(), i);
        }
        KNOWN_ZONES = new HashMap<String, String>(30);
        KNOWN_ZONES.put("UT", "+0000");
        KNOWN_ZONES.put("GMT", "+0000");
        KNOWN_ZONES.put("EDT", "-0400");
        KNOWN_ZONES.put("EST", "-0500");
        KNOWN_ZONES.put("CDT", "-0500");
        KNOWN_ZONES.put("CST", "-0600");
        KNOWN_ZONES.put("MDT", "-0600");
        KNOWN_ZONES.put("MST", "-0700");
        KNOWN_ZONES.put("PDT", "-0700");
        KNOWN_ZONES.put("PST", "-0800");
        KNOWN_ZONES.put("A", "+0100");
        KNOWN_ZONES.put("B", "+0200");
        KNOWN_ZONES.put("C", "+0300");
        KNOWN_ZONES.put("D", "+0400");
        KNOWN_ZONES.put("E", "+0500");
        KNOWN_ZONES.put("F", "+0600");
        KNOWN_ZONES.put("G", "+0700");
        KNOWN_ZONES.put("H", "+0800");
        KNOWN_ZONES.put("I", "+0900");
        KNOWN_ZONES.put("K", "+1000");
        KNOWN_ZONES.put("L", "+1100");
        KNOWN_ZONES.put("M", "+1200");
        KNOWN_ZONES.put("N", "-0100");
        KNOWN_ZONES.put("O", "-0200");
        KNOWN_ZONES.put("P", "-0300");
        KNOWN_ZONES.put("Q", "-0400");
        KNOWN_ZONES.put("R", "-0500");
        KNOWN_ZONES.put("S", "-0600");
        KNOWN_ZONES.put("T", "-0700");
        KNOWN_ZONES.put("U", "-0800");
        KNOWN_ZONES.put("V", "-0900");
        KNOWN_ZONES.put("W", "-1000");
        KNOWN_ZONES.put("X", "-1100");
        KNOWN_ZONES.put("Y", "-1200");
        KNOWN_ZONES.put("Z", "+0000");
        sAbsYFirstPattern = Pattern.compile(ABSDATE_YFIRST_PATTERN);
        sAbsYLastPattern = Pattern.compile(ABSDATE_YLAST_PATTERN);
        sRelDatePattern = Pattern.compile(RELDATE_PATTERN);
        sAbsMillisecsDatePattern = Pattern.compile(ABS_MILLISECS_PATTERN);
    }
}

